<?php

namespace Move\Providers;

use League\Container\Argument\RawArgument;
use League\Container\ServiceProvider\AbstractServiceProvider;
use League\Container\ServiceProvider\BootableServiceProviderInterface;
use Monolog\Formatter\FormatterInterface;
use Monolog\Logger;
use Monolog\Processor\IntrospectionProcessor;
use Monolog\Processor\MemoryPeakUsageProcessor;
use Monolog\Processor\WebProcessor;
use Move\Log\ErrorLogHandlerFactory;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerInterface;

/**
 * Class LoggerProvider
 * @package Move\Services
 */
class LoggerProvider extends AbstractServiceProvider implements BootableServiceProviderInterface
{
    /**
     * @var array
     */
    protected $provides = [
        LoggerInterface::class,
    ];

    /**
     * @var string
     */
    private $appName;

    /**
     * @var array
     */
    private $handlers;

    /**
     * @var int
     */
    private $loggerLevel;

    /**
     * @var FormatterInterface
     */
    private $lineFormatter;

    /**
     * LoggerProvider constructor.
     * @param string $appName
     * @param array $handlers
     * @param int $loggerLevel
     * @param FormatterInterface $lineFormater
     */
    public function __construct(
        string $appName,
        array $handlers = [],
        $loggerLevel = null,
        FormatterInterface $lineFormater = null
    ) {
        $this->appName = $appName;
        $this->handlers = $handlers;
        $this->loggerLevel = $loggerLevel;
        $this->lineFormatter = $lineFormater;
    }

    /**
     * @inheritdoc
     */
    public function boot()
    {
        $this->getContainer()
            ->inflector(LoggerAwareInterface::class)
            ->invokeMethod('setLogger', [LoggerInterface::class]);
    }

    /**
     * @inheritdoc
     */
    public function register()
    {
        $logHandler = ErrorLogHandlerFactory::create($this->loggerLevel);
        if ($this->lineFormatter) {
            $logHandler->setFormatter($this->lineFormatter);
        }

        $handlers = $this->handlers;
        array_unshift($handlers, $logHandler);

        $this->getContainer()
            ->share(LoggerInterface::class, Logger::class)
            ->withArguments([
                new RawArgument($this->appName),
                new RawArgument($handlers),
                new RawArgument([
                    new IntrospectionProcessor(),
                    new MemoryPeakUsageProcessor(),
                    new WebProcessor(),
                ]),
            ]);
    }

    /**
     * @param int $loggerLevel
     * @return $this
     */
    public function setLoggerLevel(int $loggerLevel)
    {
        $this->loggerLevel = $loggerLevel;
        return $this;
    }

    /**
     * @param $handler
     * @return $this
     */
    public function pushHandler($handler)
    {
        $this->handlers[] = $handler;
        return $this;
    }

    /**
     * @param FormatterInterface $lineFormatter
     * @return $this
     */
    public function setLineFormatter(FormatterInterface $lineFormatter)
    {
        $this->lineFormatter = $lineFormatter;
        return $this;
    }
}
