<?php


namespace Cms\Client;

use GuzzleHttp\Promise\EachPromise;
use Move\Http\Client\Exception\HttpClientException;
use Move\Http\Client\Exception\HttpRequestException;
use Move\Http\Client\HttpRequestOptions;
use Move\Iterator\MapIterator;
use Move\Restful\AbstractRestfulClient;
use POM\DomainObjectInterface;
use Psr\Http\Message\ResponseInterface;

/**
 * Class AbstractClient
 * @package Cms\Client
 */
abstract class AbstractClient extends AbstractRestfulClient
{

    /** @var  string */
    protected $basePath;

    /** @var  string */
    protected $fetchStrategy;

    /** @var bool */
    private $dryRun = false;

    /**
     * @return bool
     */
    public function isDryRun() : bool
    {
        return $this->dryRun;
    }

    /**
     * @param bool $dryRun
     * @return $this
     */
    public function setDryRun(bool $dryRun)
    {
        $this->dryRun = $dryRun;
        return $this;
    }

    /**
     * @return ClientIndexQueryBuilder
     */
    public function getBuilder()
    {
        $builder = new ClientIndexQueryBuilder($this);
        return $builder;
    }

    /**
     * @param string $strategy
     * @return $this
     */
    public function setIndexFetchStrategy($strategy)
    {
        $this->fetchStrategy = $strategy;
        return $this;
    }

    /**
     * @param ClientIndexQueryBuilder $queryBuilder
     * @return CmsApiIndexIterator
     */
    final public function getIndexWithBuilder(ClientIndexQueryBuilder $queryBuilder)
    {
        $queryParams = $queryBuilder->getQueryParams();
        $results = $this->getIndexWithParams($queryParams);
        return $results;
    }

    /**
     * @param string|null $documentId
     * @param \string|null $modifiedSince
     * @param bool $forget
     * @return int|bool
     * @throws \InvalidArgumentException
     */
    public function refreshIndex($documentId = null, $modifiedSince = null, bool $forget = false)
    {
        if (!$documentId && !$modifiedSince) {
            throw new \InvalidArgumentException('documentId or modifiedSince must be set');
        }
        $options = [
            HttpRequestOptions::QUERY_PARAMS => [
                'docid' => $documentId,
                'modified_since' => $modifiedSince ?: null,
            ],
        ];

        try {
            $reqMethod = $forget ? 'DELETE' : 'GET';
            $response = $this->request($reqMethod, [$this->basePath, '_index'], $options);
            return $response['indexed'] ?? 0;
        } catch (HttpClientException $e) {
            throw new \InvalidArgumentException('failed to index due to an error : ' . $e->getMessage(), null, $e);
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * @param int $id
     * @param array $queryParams
     * @param bool $nocache
     * @return null|\POM\DomainObjectInterface
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function fetchItem($id, array $queryParams = [], bool $nocache = false)
    {
        if ($nocache === true) {
            $queryParams['nocache'] = $nocache;
        }
        $options = [
            HttpRequestOptions::QUERY_PARAMS => $queryParams,
        ];
        // récuperation des données
        $dataset = $this->request('GET', [$this->basePath, $id], $options);
        if (empty($dataset) || !\is_array($dataset)) {
            return null;
        }
        // chargement données
        $dataset = $this->convertApiArray($dataset);
        return $this->handleData($dataset);
    }

    /**
     * @param DomainObjectInterface $object
     * @param string $primaryKey
     * @return bool|null|DomainObjectInterface
     * @throws \LogicException
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \InvalidArgumentException
     * @throws \UnexpectedValueException
     */
    public function saveItem(DomainObjectInterface &$object, $primaryKey = 'id')
    {
        if (!$this->basePath) {
            throw new \UnexpectedValueException('basePath attribute must be defined');
        }

        /*var_dump([
            'url' => $this->basePath,
            'opts' => [
                HttpRequestOptions::QUERY_PARAMS => ['dryrun' => $this->isDryRun()],
                HttpRequestOptions::BODY_CONTENT => array_merge(
                    $this->convertObjectToArray(
                        $object,
                        $this->createPolicy
                    ),
                    []
                ),
            ]
        ]);*/

        // check si update/create
        if (empty($object[$primaryKey])) {
            $result = $this->createObjectFromClient(
                $object,
                $this->basePath,
                [],
                ['dryrun' => $this->isDryRun()]
            );
        } else {
            $result = $this->updateObjectFromClient(
                $object,
                $this->basePath,
                $primaryKey,
                ['dryrun' => $this->isDryRun()]
            );
        }
        // mise a jour de l'objet
        if ($result instanceof DomainObjectInterface) {
            $object = $result;
        }
        return $result;
    }

    /**
     * @param DomainObjectInterface $object
     * @param string $primaryKey
     * @return bool
     * @throws \LogicException
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \InvalidArgumentException
     * @throws \UnexpectedValueException
     */
    public function deleteItem(DomainObjectInterface $object, $primaryKey = 'id')
    {
        if (!$this->basePath) {
            throw new \UnexpectedValueException('basePath attribute must be defined');
        }
        $result = $this->deleteObjectFromClient(
            $object,
            $this->basePath,
            $primaryKey,
            ['dryrun' => $this->isDryRun()]
        );
        return $result;
    }

    /**
     * @param array $dataset
     * @return null|DomainObjectInterface
     */
    public function callbackIterator($dataset)
    {
        if (\is_array($dataset)) {
            $dataset = $this->convertApiArray($dataset);
            return $this->handleData($dataset);
        }
        return null;
    }

    /**
     * @return string
     */
    public function getBasePath() : string
    {
        return $this->basePath;
    }

    /**
     * @param array|string $segments
     * @param array $queryParams
     * @return CmsApiIndexIterator
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \InvalidArgumentException
     * @throws \LogicException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \UnexpectedValueException
     */
    protected function loadIndexFromClient($segments, array $queryParams = []) : CmsApiIndexIterator
    {
        $segments = (array)$segments;
        if ($this->fetchStrategy === ClientFetchStrategy::FETCH_ASYNC) {
            array_unshift($segments, '2.0');
        }

        // récuperation des données
        $dataset = $this->request('GET', (array)$segments, [
            HttpRequestOptions::QUERY_PARAMS => $queryParams,
        ]);

        $iterator = $this->handleIndexData($dataset);
        if (!$iterator instanceof CmsApiIndexIterator) {
            throw new \UnexpectedValueException('index data handler must return a CmsApiIndexIterator');
        }
        return $iterator;
    }

    /**
     * @param iterable $ids
     * @param array $queryParams
     * @param null|int $concurrency
     * @return DomainObjectInterface[]
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     * @throws \LogicException
     */
    public function fetchBulkItems($ids, array $queryParams = [], int $concurrency = null) : array
    {
        if (empty($ids)) {
            return [];
        }

        // generation de la liste de promise
        $promises = (function ($ids, $queryParams) {
            $requestOpts = [
                HttpRequestOptions::QUERY_PARAMS => $queryParams,
            ];
            foreach ($ids as $id) {
                if (!is_numeric($id) && !\is_string($id)) {
                    throw new \InvalidArgumentException('id must be scalar : ' . $id . ' (' . gettype($id) . ')');
                }
                yield $this
                    ->getClient()
                    ->requestAsync('GET', [$this->basePath, $id], $requestOpts)
                    ->then(function (ResponseInterface $response) {
                        if ($_GET['debug']) {
                            dump($response);
                        }
                        $dataset = $this->getDatasetFromResponse($response);
                        if ($_GET['debug']) {
                            dump($dataset);
                        }
                        return $dataset;
                    });
            }
        })($ids, $queryParams);

        // declanche la promise par lot de concurrency
        $results = [];
        (new EachPromise($promises, [
            'concurrency' => $concurrency ?: 5,
            'fulfilled' => function ($data, $idx) use (&$results) {
                if (!empty($data)) {
                    $results[$idx] = $data;
                }
            },
            'rejected' => function ($reason) {
                if ($reason instanceof HttpClientException) {
                    $this->handleHttpClientException($reason);
                } elseif ($reason instanceof HttpRequestException) {
                    $this->handleHttpRequestException($reason);
                }
            },
        ]))->promise()->wait();

        // trie des resultats pour respect de l'ordre d'arrivé
        ksort($results);

        // transormation des resultats de l'api
        $results = array_map([$this, 'callbackIterator'], $results);
        $results = array_filter($results, function ($value) {
            return null !== $value;
        });

        return $results;
    }

    /**
     * @param array $dataset
     * @return \Cms\Client\CmsApiIndexIterator
     * @throws \LogicException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     */
    protected function handleIndexData(array $dataset = [])
    {
        // creation de l'iterateur de réponse
        $iterator = new \ArrayIterator($dataset['data'] ?? []);

        // convertion des resultats id seul en dataset correct
        if ($this->fetchStrategy === ClientFetchStrategy::FETCH_ASYNC) {
            $results = $this->fetchBulkItems((function ($iterator) {
                foreach ($iterator as $data) {
                    yield $data['id'];
                }
            })($iterator));
            $iterator = new \ArrayIterator($results);
        } else {
            $iterator = new MapIterator($iterator, [$this, 'callbackIterator']);
            $iterator = new \CallbackFilterIterator($iterator, function ($value) {
                return null !== $value;
            });
        }

        // récup des meta données
        $offset = $total = 0;
        if (!empty($dataset['meta']['total'])) {
            $total = $dataset['meta']['total'];
        }
        if (!empty($dataset['meta']['offset'])) {
            $offset = $dataset['meta']['offset'];
        }

        // iterator spécifique
        $iterator = new CmsApiIndexIterator($iterator, $offset, $total, 20);
        return $iterator;
    }

    /**
     * @param array $dataset
     * @return DomainObjectInterface
     */
    abstract public function handleData($dataset = []);

    /**
     * @param array $queryParams
     * @return CmsApiIndexIterator
     */
    abstract public function getIndexWithParams($queryParams);
}
