<?php

namespace Move\Http\Route;

/**
 * Class RouteRegistrer
 * @package Move\Http\Route
 */
class RouteRegistrer
{
    /**
     * @var string
     */
    protected $patternForId = ':number';

    /**
     * @var RouterAdapterInterface
     */
    private $router;

    /**
     * @var string
     */
    private $ressource_prefix = '/';


    /**
     * RouteRegistrer constructor.
     * @param RouterAdapterInterface $router
     */
    public function __construct(RouterAdapterInterface $router)
    {
        $this->router = $router;
    }

    /**
     * @return RouterAdapterInterface
     */
    public function getRouter()
    {
        return $this->router;
    }

    /**
     * Ajoute un prefix generique à toutes les ressources
     * @param string $prefix
     */
    public function setRessourcePrefix($prefix)
    {
        $this->ressource_prefix = sprintf('/%s', trim($prefix, '/'));
    }

    /**
     * assigne une ressource au routeur
     * le prefix désigne la base d'url, le handler désigne la classe du controller
     * on ajoute les méthodes suivantes dans le router :
     * GET /prefix > handler::index
     * GET /prefix/{id} > handler::show
     * POST /prefix > handler::create
     * DELETE /prefix/{id} > handler::delete
     * PUT /prefix > handler::create
     * PUT /prefix/{id} > handler::update
     * @param string $prefix
     * @param string $handler
     * @param array $opts
     * @return int|mixed
     */
    public function ressource($prefix, $handler, array $opts = [])
    {
        $prefix = trim($prefix, '/');

        // selection de la strategie
        $itemName = 'id';
        if (!empty($opts['item_name']) && !is_numeric($opts['item_name'])) {
            $itemName = $opts['item_name'];
        }

        // liste methodes / verbs
        $methodList = [
            'GET' => [$itemName => 'show', 'index'],
            'POST' => ['create'],
            'DELETE' => [$itemName => 'delete'],
            'PUT' => [$itemName => 'update', 'create'],
            'OPTIONS' => [$itemName => 'options', 'options'],
        ];

        // selection su pattern pour l'id
        $idPattern = $this->patternForId;
        if (!empty($opts['id_pattern']) && preg_match('@^:@', $opts['id_pattern'])) {
            $idPattern = $opts['id_pattern'];
        }

        // selection des verbs
        if (!empty($opts['only_verbs'])) {
            $methodList = array_intersect_key($methodList, array_flip($opts['only_verbs']));
        }

        // selection des methodes
        if (!empty($opts['only'])) {
            foreach ($methodList as &$methodGroup) {
                $methodGroup = array_intersect($methodGroup, $opts['only']);
            }
            unset($methodGroup);
        }

        // check si parent existe
        $groupPrefix = $this->ressource_prefix;
        if (!empty($opts['nested'])) {
            $groupPrefix .= sprintf('/%s', ltrim($opts['nested'], '/'));
        }

        // ajout de l'extention authorisé
        $suffixList = [];
        if (!empty($opts['extensions']) && \is_array($opts['extensions'])) {
            $suffixList = array_filter($opts['extensions'], function ($ext) {
                return preg_match('@^\.[a-z0-9]+$@i', $ext);
            });
        }
        $suffixList[] = '';

        // generation de la liste de route
        $routeList = [];
        foreach ($methodList as $verb => $verbGroup) {
            foreach ($verbGroup as $withItem => $methodName) {
                foreach ($suffixList as $suffix) {
                    $routeList[] = [
                        $verb,
                        $prefix . (!is_numeric($withItem) ? '/{' . $withItem . $idPattern . '}' : '') . $suffix,
                        $handler . '::' . $methodName,
                    ];
                }
            }
        }

        if (!$routeList) {
            return null;
        }

        // creation des goupes de routes
        if (empty($opts['apply_now'])) {
            return $this->router->group($groupPrefix, $routeList);
        }
        foreach ($routeList as $routeInfo) {
            $this->router->map(...$routeInfo);
        }

        return count($routeList);
    }

    /**
     * assign un controller au routeur
     * le prefix désigne la base de l'url, le handler désigne la classe du controller
     * on ajoute les méthodes correspondant a : GET /prefix/{action}
     * @param string $prefix
     * @param string $handler
     * @param array $opts
     * @return mixed
     */
    public function controller($prefix, $handler, array $opts = [])
    {
        $methodList = [];

        // selection de l'action par default (la racine)
        $defaultAction = 'index';
        if (!empty($opts['default_action'])) {
            $defaultAction = $opts['default_action'];
        }

        // check si parent existe
        $groupPrefix = '/';
        if (!empty($opts['nested'])) {
            $groupPrefix = sprintf('/%s', ltrim($opts['nested'], '/'));
        }

        // selection des methodes
        if (!empty($opts['only'])) {
            $methodList = $opts['only'];
        }

        // fabrication de la liste des actions
        $routeList = [];
        foreach ($methodList as $methodName) {
            $routeList[] = [
                'GET',
                $prefix . ($methodName !== $defaultAction ? '/' . $methodName : ''),
                $handler . '::' . $methodName,
            ];
        }

        //var_dump($prefix, $routeList);
        // creation des goupes de routes
        return $this->router->group($groupPrefix, $routeList);
    }

    /**
     * @return string
     */
    public function getPatternForId()
    {
        return $this->patternForId;
    }

    /**
     * @param string $patternForId
     */
    public function setPatternForId($patternForId)
    {
        $this->patternForId = $patternForId;
    }
}
