<?php

namespace Move\Search;

use Elastica\Client;
use Elastica\Index;
use Move\Environment;
use Move\Search\Analyzer\ElasticAnalyzer;

/**
 * Class ElasticIndexBuilder
 * @package Move\Search
 */
class ElasticIndexBuilder implements ElasticIndexBuilderInterface
{

    /** @var  string */
    protected $indexName;

    /** @var  array */
    protected $aliases = [];

    /** @var  \Move\Search\ElasticMappingBuilderInterface[] */
    protected $mappings = [];

    /** @var array */
    protected $args;

    /** @var  array */
    protected $opts = [];

    /**
     * ElasticIndexBuilder constructor.
     * @param string $indexName
     * @param int $shards
     * @param int $reps
     * @param array|null $analyzers
     */
    public function __construct(
        string $indexName,
        int $shards,
        int $reps = 2,
        array $analyzers = null
    ) {
        $this->indexName = $indexName;
        $this->args = [
            'max_result_window' => 50000,
            'number_of_shards' => $shards,
            'number_of_replicas' => (new Environment())->isProd() ? $reps : 0,
            'analysis' => [
                'analyzer' => [],
                'filter' => [],
            ],
        ];

        if ($analyzers !== null) {
            foreach ($analyzers as $item) {
                $this->setAnalizer($item);
            }
        }
    }

    /**
     * @param int $max
     * @return $this
     */
    public function setNumOfReplicas(int $max) : self
    {
        $this->opts = ['number_of_replicas' => $max];
        return $this;
    }

    /**
     * @param int $max
     * @return $this
     */
    public function setMaxResultWindow(int $max) : self
    {
        $this->opts = ['max_result_window' => $max];
        return $this;
    }

    /**
     * @return $this
     */
    public function forceRecreate() : self
    {
        $this->opts = ['recreate' => true];
        return $this;
    }

    /**
     * @param \Elastica\Index $index
     * @param array|null $aliases
     * @return \Elastica\Index
     * @throws \RuntimeException
     */
    public function createAliases(Index $index, array $aliases = null) : Index
    {
        if (!$index->exists()) {
            throw new \RuntimeException('index ' . $this->indexName . ' not exists');
        }

        $aliases = $aliases ?: $this->aliases;
        foreach ($aliases as $alias) {
            if (!$index->hasAlias($alias)) {
                $index->addAlias($alias, true);
            }
        }

        return $index;
    }

    /**
     * @param \Elastica\Client $client
     * @param array|null $aliases
     * @param \Move\Search\ElasticMappingBuilderInterface[]|null $mappings
     * @return \Elastica\Index
     * @throws \RuntimeException
     * @throws \Elastica\Exception\InvalidException
     * @throws \Elastica\Exception\ResponseException
     * @throws \InvalidArgumentException
     */
    public function createIndex(
        Client $client,
        array $aliases = null,
        array $mappings = null
    ) : Index {
        $index = $client->getIndex($this->indexName);
        if (empty($this->opts['recreate']) && $index->exists()) {
            throw new \RuntimeException('index ' . $this->indexName . ' already exists');
        }
        $index->create($this->args, $this->opts);

        $mappings = $mappings ?: $this->mappings;
        foreach ($mappings as $mapping) {
            if (!$mapping instanceof ElasticMappingBuilderInterface) {
                throw new \InvalidArgumentException('invalid mapping');
            }
        }
        
        foreach ($mappings as $mapping) {
            $mapping->createMapping($index);
        }

        // creation des alias d'index
        return $this->createAliases($index, $aliases);
    }

    /**
     * @param string $indexName
     * @return $this
     */
    public function setIndexName(string $indexName) : ElasticIndexBuilderInterface
    {
        $this->indexName = $indexName;
        return $this;
    }

    /**
     * @param array $aliases
     * @return $this
     */
    public function setAliases(array $aliases) : ElasticIndexBuilderInterface
    {
        $this->aliases = $aliases;
        return $this;
    }

    /**
     * @param \Move\Search\ElasticMappingBuilderInterface $mappingBuilder
     * @return $this
     */
    public function setMappingBuilder(ElasticMappingBuilderInterface $mappingBuilder) : ElasticIndexBuilderInterface
    {
        $this->mappings[] = $mappingBuilder;
        return $this;
    }

    /**
     * @param \Move\Search\Analyzer\ElasticAnalyzer $analyzer
     * @return $this
     */
    public function setAnalizer(ElasticAnalyzer $analyzer) : ElasticIndexBuilderInterface
    {
        $this->args['analysis']['filter'] = array_merge(
            $this->args['analysis']['filter'],
            $analyzer->getFilters()
        );

        $this->args['analysis']['analyzer'][$analyzer->getName()] = [
            'tokenizer' => $analyzer->getTokenizer(),
            'char_filter' => $analyzer->getCharFilterNames(),
            'filter' => $analyzer->getFilterNames(),
        ];

        return $this;
    }

    /**
     * @return string
     */
    public function getIndexName() : string
    {
        return $this->indexName;
    }
}
