<?php

namespace Cms\Extension\Component;

use Cms\Cache\CacheHelperTrait;
use Cms\Client\Service\ArticleService;
use League\Plates\Template\Template;
use Move\Cache\CacheAwareInterface;
use Move\Iterator\MapIterator;
use Move\Template\Component\AbstractComponent;
use Psr\Cache\CacheItemInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;

/**
 * Class AbstractPopularComponent
 * @package Cms\Extension\Component
 */
abstract class AbstractPopularComponent extends AbstractComponent implements
    CacheAwareInterface,
    LoggerAwareInterface
{
    use LoggerAwareTrait;
    use CacheHelperTrait;

    /** @var string */
    protected $functionName = 'popular';

    /** @var  ArticleService */
    protected $articleService;

    /** @var string */
    private $keyFileLocation;

    /** @var string */
    private $appName;

    /** @var string */
    private $viewId;

    /**
     * PopularComponent constructor.
     * Use the developers console and download your service account
     * credentials in JSON format.
     * @param ArticleService $articleService
     * @param string $keyFileLocation
     * @param string $appName
     * @param string $viewId
     * @param \Psr\Log\LoggerInterface|null $logger
     */
    public function __construct(
        ArticleService $articleService,
        $keyFileLocation,
        $appName,
        $viewId,
        LoggerInterface $logger = null
    ) {
        $this->articleService = $articleService;
        $this->keyFileLocation = $keyFileLocation;
        $this->appName = $appName;
        $this->viewId = $viewId;
        $this->logger = $logger ?? new NullLogger();
    }

    /**
     * @param Template $template
     * @param array $data
     */
    protected function assignData(Template $template, $data = [])
    {
        $cacheKey = $this->appName . '^popular^2';
        $templateData = [];

        // check si donnée en cache
        if (!$this->hitFromCache($cacheKey, 3600, $cacheItem)) {
            $analytics = $this->initializeAnalytics();
            $reports = $this->getReport($analytics);
            $mostRead = $this->generateResults($reports);
            // crée un iterateur
            $mostReadIterator = new \ArrayIterator($mostRead['ga:pagePath'] ?? []);
            $articlesIterator = new MapIterator($mostReadIterator, [$this, 'getArticleModel']);
            $articles = new \CallbackFilterIterator(
                $articlesIterator,
                [$this, 'isArticleRestfulModel']
            );
            $articles = new \CallbackFilterIterator(
                $articles,
                $this->isUniqueArticle()
            );
            $articles = new \LimitIterator($articles, 0, 5);

            $templateData['articles'] = iterator_to_array($articles);

            // charge les données dans l'item de cache
            $this->setInCache($cacheItem, $templateData);
        } // récuperation des données dans le cache
        elseif ($cacheItem instanceof CacheItemInterface) {
            $templateData = $cacheItem->get();
        }
        $template->data(array_merge($templateData, $data));
    }

    /**
     * @return \Google_Service_AnalyticsReporting
     */
    protected function initializeAnalytics() : \Google_Service_AnalyticsReporting
    {
        // Create and configure a new client object.
        $client = new \Google_Client();
        $client->setLogger($this->logger);
        $client->setApplicationName($this->appName);
        $client->setAuthConfig($this->keyFileLocation);
        $client->setScopes(['https://www.googleapis.com/auth/analytics.readonly']);
        $analytics = new \Google_Service_AnalyticsReporting($client);
        return $analytics;
    }

    /**
     * @return \Google_Service_AnalyticsReporting_ReportRequest
     */
    protected function prepareRequest() : \Google_Service_AnalyticsReporting_ReportRequest
    {
        // Create the DateRange object.
        $dateRange = new \Google_Service_AnalyticsReporting_DateRange();
        $dateRange->setStartDate('7daysAgo');
        $dateRange->setEndDate('today');

        // Create the Metrics object.
        $metric = new \Google_Service_AnalyticsReporting_Metric();
        $metric->setExpression('ga:pageviews');
        $metric->setAlias('Page vues');

        // Order By
        $orderby = new \Google_Service_AnalyticsReporting_OrderBy();
        $orderby->setFieldName('ga:pageviews');
        $orderby->setOrderType('VALUE');
        $orderby->setSortOrder('DESCENDING');

        // Dimensions
        $dimensionPath = new \Google_Service_AnalyticsReporting_Dimension();
        $dimensionPath->setName('ga:pagePath');
        $dimensionTitle = new \Google_Service_AnalyticsReporting_Dimension();
        $dimensionTitle->setName('ga:pageTitle');

        // Create the ReportRequest object.
        $request = new \Google_Service_AnalyticsReporting_ReportRequest();
        $request->setViewId($this->viewId);
        $request->setDateRanges($dateRange);
        $request->setMetrics([$metric]);
        $request->setDimensions([$dimensionPath, $dimensionTitle]);
        $request->setOrderBys([$orderby]);
        $request->setPageSize(50);

        return $request;
    }

    /**
     * @param \Google_Service_AnalyticsReporting $analytics
     * @return \Google_Service_AnalyticsReporting_GetReportsResponse
     */
    protected function getReport(&$analytics)
    {
        $request = $this->prepareRequest();
        $body = new \Google_Service_AnalyticsReporting_GetReportsRequest();
        $body->setReportRequests([$request]);
        return $analytics->reports->batchGet($body);
    }

    /**
     * @param \Google_Service_AnalyticsReporting_GetReportsResponse $reports
     * @return array
     */
    protected function generateResults(&$reports)
    {
        $mostRead = [];
        $countReports = count($reports);
        for ($reportIndex = 0; $reportIndex < $countReports; $reportIndex++) {
            $report = $reports[$reportIndex];
            $header = $report->getColumnHeader();
            $dimensionHeaders = $header->getDimensions();
            $metricHeaders = $header->getMetricHeader()->getMetricHeaderEntries();
            $rows = $report->getData()->getRows();

            for ($rowIndex = 0, $rowCount = count($rows); $rowIndex < $rowCount; $rowIndex++) {
                $row = $rows[$rowIndex];
                $dimensions = $row->getDimensions();
                $metrics = $row->getMetrics();
                for ($i = 0; $i < count($dimensionHeaders) && $i < count($dimensions); $i++) {
                    $mostRead[$dimensionHeaders[$i]][] = $dimensions[$i];
                }

                for ($j = 0; $j < count($metricHeaders) && $j < count($metrics); $j++) {
                    $entry = $metricHeaders[$j];
                    $values = $metrics[$j];
                    $valuesCount = count($values->getValues());
                    for ($valueIndex = 0; $valueIndex < $valuesCount; $valueIndex++) {
                        $value = $values->getValues()[$valueIndex];
                        $mostRead[$entry->getName()] = $value;
                    }
                }
            }
        }
        return $mostRead;
    }

    /**
     * @param array $current
     * @return bool
     */
    public function isArticleRestfulModel($current)
    {
        return (!empty($current['id']));
    }

    /**
     * @return \Closure
     */
    public function isUniqueArticle() : callable
    {
        $idsMap = [];
        return function ($current) use (&$idsMap) {
            if (!\in_array($current['id'], $idsMap)) {
                $idsMap[] = $current['id'];
                return true;
            }
            return false;
        };
    }

    /**
     * @param string $articleUrl
     * @return array
     */
    abstract public function getArticleModel($articleUrl);
}
