<?php

namespace Move\Specification;

use PHPUnit\Framework\TestCase;

/**
 * Class LengthSpecificationTest
 * @package Move\Specification
 */
class LengthSpecificationTest extends TestCase
{
    /**
     * @return array
     */
    public static function provideIsSatisfiedBy(){
        $number = 4;
        $string = 'qwer';
        $array = [1, 2, 3, 4];

        return [
            'Array' => [$array, $number],
            'String' => [$string, $number],
            'number' => [$number, $number],
        ];
    }

    /**
     * @param $object
     * @param $condition
     * @param $length
     * @param $expected
     */
    public function singleTest($object, $condition, $length, $expected)
    {
        $lengthSpec = new LengthSpecification($condition, $length);
        $this->assertEquals($expected, $lengthSpec->isSatisfiedBy($object));
    }

    /**
     * @param mixed $object
     * @param mixed $condition
     * @param mixed $value
     * @param mixed $expected
     */
    public function checkPossibilities($object, $condition, $value, $expected)
    {
        $this->singleTest($object, $condition, $value - 1, $expected['bigger']);
        $this->singleTest($object, $condition, $value ,  $expected['equals']);
        $this->singleTest($object, $condition, $value + 1,  $expected['smaller']);
    }

    /**
     * @covers LengthSpecification::__construct
     * @return LengthSpecification
     */
    public function testConstruct()
    {
        $lengthSpec = new LengthSpecification(
            LengthSpecification::BIGGER_EQUAL_THAN,
            1
        );
        $this->assertInstanceOf(LengthSpecification::class, $lengthSpec);
        return $lengthSpec;
    }

    /**
     * @depends testConstruct
     * @dataProvider provideIsSatisfiedBy
     * @param $object
     * @param $length
     */
    public function testIsSatisfiedBy($object, $length)
    {
        $this->checkPossibilities($object, LengthSpecification::BIGGER_EQUAL_THAN, $length, [
            'bigger' => true,
            'equals' => true,
            'smaller' => false
        ]);

        $this->checkPossibilities($object, LengthSpecification::BIGGER_THAN, $length, [
            'bigger' => true,
            'equals' => false,
            'smaller' => false
        ]);

        $this->checkPossibilities($object, LengthSpecification::LESS_EQUAL_THAN, $length, [
            'bigger' => false,
            'equals' => true,
            'smaller' => true
        ]);

        $this->checkPossibilities($object, LengthSpecification::LESS_THAN, $length, [
            'bigger' => false,
            'equals' => false,
            'smaller' => true
        ]);
    }

    public function testUnknownFormat()
    {
        $this->checkPossibilities((object) [], LengthSpecification::LESS_THAN, 1, [
            'bigger' => false,
            'equals' => false,
            'smaller' => false
        ]);
    }
}
