<?php


namespace Cms\Restful;

use League\Route\Http\Exception\BadRequestException;
use Move\Http\ResponseUtils;
use Move\Http\Strategy\ResponseFactory;
use Move\Iterator\MapIterator;
use POM\DomainObjectInterface;
use Psr\Http\Message\ServerRequestInterface;
use Zend\Diactoros\Response\EmptyResponse;

/**
 * Class ResponseControllerTrait
 * @package Cms\Restful
 */
trait ResponseControllerTrait
{

    /** @var int */
    protected $defaultItemByPage = 20;

    /**
     * @return callable
     */
    abstract protected function getTransformer();


    /**
     * @param array|\Iterator $collection
     * @param int $pageNum
     * @param int $byPage
     * @return array|\Iterator
     * @throws \League\Route\Http\Exception\BadRequestException
     * @throws \UnexpectedValueException
     */
    protected function getLimitIterator($collection, $pageNum = 0, $byPage = null)
    {
        if (empty($collection)) {
            return [];
        }
        if (!$collection instanceof \Iterator) {
            if (!\is_array($collection)) {
                throw new \UnexpectedValueException('collection must be array or iterator');
            }
            $collection = new \ArrayIterator($collection);
        }

        // check
        if ($pageNum >= 0) {
            if ($collection instanceof \Countable) {
                $totalResult = count($collection);
            } else {
                $totalResult = iterator_count($collection);
            }
            $byPage = $byPage ?: $this->defaultItemByPage;
            if ($totalResult < $pageNum * $byPage) {
                throw new BadRequestException('page out of bounds');
            }
            $collection = new \LimitIterator($collection, $pageNum * $byPage, $byPage);
        }

        return $collection;
    }

    /**
     * @param mixed $object
     * @return int
     */
    private function countObject($object)
    {
        return \is_array($object) || $object instanceof \Countable ? count($object) :
            ($object instanceof \Iterator ? iterator_count($object) : 0);
    }

    /**
     * @param array|\Iterator $collection
     * @return int
     * @throws \InvalidArgumentException
     */
    private function getCollectionCount($collection)
    {
        if (!$collection instanceof \Iterator && !\is_array($collection)) {
            throw new \InvalidArgumentException('collection must be array or iterator');
        }
        if ($collection instanceof \AppendIterator) {
            $totalResult = 0;
            foreach ($collection->getArrayIterator() as $iterator) {
                $totalResult += $this->countObject($iterator);
            }
        } else {
            $totalResult = $this->countObject($collection);
        }
        return $totalResult;
    }

    /**
     * @param int|null $defaultTTL
     * @param int|null $archiveTTL
     */
    protected function getCollectionModifiedHeaders()
    {
        // TODO
    }

    /**
     * @param array|\Iterator $collection
     * @param int $pageNum
     * @param int $byPage
     * @return \Psr\Http\Message\ResponseInterface|ResponseFactory
     * @throws \InvalidArgumentException
     * @throws \UnexpectedValueException
     * @throws BadRequestException
     */
    protected function makeCollectionResponse($collection, $pageNum = 0, $byPage = null)
    {
        // check total
        $totalResult = $this->getCollectionCount($collection);
        if (0 === $totalResult) {
            $response = new EmptyResponse();
            $response = ResponseUtils::withLastModified($response, time(), 300);
            return $response;
        }

        // check transformer
        $transformer = $this->getTransformer();
        if (!\is_callable($transformer)) {
            throw new \UnexpectedValueException('getTransformer must be callable');
        }

        // cas des transformer avec mode index
        if ($transformer instanceof TransformerStrategyAwareInterface) {
            $strategy = TransformStrategy::PRIMARY_KEYS;
            if (isset($this->collectionStrategy)) {
                $strategy = $this->collectionStrategy;
            }
            if (!empty($strategy)) {
                $transformer->setStrategy($strategy);
            }
        }

        // map des resultat
        $byPage = $byPage ?: $this->defaultItemByPage;
        $result = $this->getLimitIterator($collection, $pageNum, $byPage);
        $result = new MapIterator($result, $transformer);

        // creation de la response
        $response = new ResponseFactory([
            'meta' => [
                'total' => $totalResult,
                'by_page' => $pageNum >= 0 ? ($byPage > 0 ? $byPage : $totalResult) : $totalResult,
                'offset' => $pageNum >= 0 ? $pageNum * $byPage : 0,
            ],
            'data' => $resultSet = iterator_to_array($result, false),
        ]);

        // ajout des header de cache
        /*$defaultTTL = isset($this->object_ttl) ? $this->object_ttl : null;
        $archiveTTL = isset($this->object_archive_ttl) ? $this->object_archive_ttl : null;
        $response = $response->mergeHeaders($this->getCollectionModifiedHeaders(
            $resultSet, $defaultTTL, $archiveTTL
        ));*/
        $response = $response->mergeHeaders([
            'Cache-Control' => ['public'],
        ]);

        return $response;
    }

    /**
     * @param ServerRequestInterface $request
     * @param DomainObjectInterface $object
     * @return bool
     */
    protected function isObjectNotModified(
        ServerRequestInterface $request,
        DomainObjectInterface $object
    ) {
        $stampProperty = 'modified_at';
        $ifModifiedGmt = $request->getHeaderLine('if-modified-since');
        if (!empty($ifModifiedGmt)
            && isset($object[$stampProperty])
            && $object[$stampProperty] instanceof \DateTime
        ) {
            /** @noinspection UsageOfSilenceOperatorInspection */
            return $object[$stampProperty]->getTimestamp() === @strtotime($ifModifiedGmt);
        }
        return false;
    }

    /**
     * @param DomainObjectInterface $object
     * @param int|null $defaultTTL
     * @param int|null $archiveTTL
     * @return array
     */
    protected function getObjectModifiedHeaders(
        DomainObjectInterface $object,
        $defaultTTL = null,
        $archiveTTL = null
    ) {
        $stampProperty = 'modified_at';
        $defaultTTL = $defaultTTL ?? $this->object_ttl ?? null;
        $archiveTTL = $archiveTTL ?? $this->object_archive_ttl ?? null;
        if (!empty($defaultTTL)) {
            if (isset($object[$stampProperty])) {
                if ($object[$stampProperty] instanceof \DateTime) {
                    $responseTTL = $defaultTTL;
                    // si object non modifié depuis longtemps
                    $diffDays = $object[$stampProperty]->diff(date_create('now'))->format('%a');
                    if ($diffDays > 30 && $archiveTTL) {
                        $responseTTL = $archiveTTL;
                    }
                    $headers = ResponseUtils::getLastModifiedHeaders(
                        $object[$stampProperty]->getTimestamp(),
                        $responseTTL
                    );
                    return $headers;
                }
            } else {
                $headers = ResponseUtils::getLastModifiedHeaders(
                    time(),
                    max($archiveTTL ?: 0, $defaultTTL)
                );
                return $headers;
            }
        }
        return [
            'Cache-Control' => ['public'],
        ];
    }

    /**
     * @param DomainObjectInterface $object
     * @param int $code
     * @return ResponseFactory
     * @throws \UnexpectedValueException
     */
    protected function makeObjectResponse(DomainObjectInterface $object, $code = 200)
    {
        // check transformer
        $transformer = $this->getTransformer();
        if (!\is_callable($transformer)) {
            throw new \UnexpectedValueException('getTransformer must be callable');
        }

        // convertion de l'objet en array
        $dataset = $transformer($object);
        $response = new ResponseFactory($dataset, $code);

        // ajout des header de cache
        if ((int)$code === 200) {
            $defaultTTL = $this->object_ttl ?? null;
            $archiveTTL = $this->object_archive_ttl ?? null;
            $response = $response->mergeHeaders($this->getObjectModifiedHeaders(
                $object,
                $defaultTTL,
                $archiveTTL
            ));
        }

        return $response;
    }
}
