<?php

namespace Cms\Import;

use Bramus\Monolog\Formatter\ColoredLineFormatter;
use Monolog\Handler\BufferHandler;
use Monolog\Handler\GroupHandler;
use Monolog\Handler\StreamHandler;
use Monolog\Logger;
use Monolog\Processor\MemoryUsageProcessor;
use Psr\Log\LoggerInterface;

/**
 * @param string $errorlog
 */
function handle_buffer_sigterm(string $errorlog)
{
    $bufferHandler = create_buffer_handler($errorlog);
    $sigterm = function () use ($bufferHandler) {
        $bufferHandler->close();
        exit;
    };
    // fonction de fermeture en cas de giterm/int
    pcntl_signal(SIGINT, $sigterm);
    pcntl_signal(SIGTERM, $sigterm);
}

/**
 * @param string $errorlog
 * @return \Monolog\Handler\BufferHandler
 */
function create_buffer_handler(string $errorlog) : BufferHandler
{
    static $bufferHandler = null;

    // config
    $bufferHandler = $bufferHandler ?: new BufferHandler(
        new StreamHandler(
            $errorlog,
            Logger::ERROR,
            true,
            0777
        )
    );

    return $bufferHandler;
}

/**
 * @param string $errorlog
 * @param int $stdoutLevel
 * @return \Psr\Log\LoggerInterface
 */
function create_logger(string $errorlog = null, int $stdoutLevel = Logger::DEBUG) : LoggerInterface
{
    // output handler
    $streamHandler = new StreamHandler('php://stdout', $stdoutLevel);
    $streamHandler->setFormatter(new ColoredLineFormatter());

    // group de handler
    $groupHandler = [
        $streamHandler,
    ];

    // config
    $logger = new Logger('import_logger');
    if ($errorlog !== null) {
        $groupHandler[] = create_buffer_handler($errorlog);
    }

    $logger->pushHandler(new GroupHandler($groupHandler));

    $logger->pushProcessor(new MemoryUsageProcessor());

    return $logger;
}
