<?php


namespace Move;

/**
 * Class Environment
 * @package Move
 */
class Environment
{
    public const DEV = 'development';

    public const TEST = 'test';

    public const STAGING = 'staging';

    public const PROD = 'production';

    /** @var array */
    protected static $ALIAS = [
        'development' => ['devel', 'dev', 'local'],
        'test' => ['testing'],
        'staging' => ['recette', 'preprod'],
        'production' => ['prod'],
    ];

    /** @var string */
    public static $DEFAULT = 'production';

    /** @var string */
    protected $environmentName;

    /** @var string */
    protected $type;

    /**
     * Environment constructor.
     * @param string $envVarName ('APP_ENV')
     */
    public function __construct(string $envVarName = null)
    {
        $this->environmentName = $envVarName ?: 'APP_ENV';
        $this->type = $this->resolvType();
    }

    /**
     * @return string
     */
    protected function resolvType() : string
    {
        if (isset($_SERVER[$this->environmentName])) {
            if (!\is_string($_SERVER[$this->environmentName])) {
                throw new \UnexpectedValueException(
                    'Env must be a string ' . gettype($_SERVER[$this->environmentName]) . 'given'
                );
            }
            return $_SERVER[$this->environmentName];
        }
        if (getenv($this->environmentName) !== false) {
            return getenv($this->environmentName);
        }
        return static::$DEFAULT;
    }

    /**
     * @param string $type
     * @return bool
     */
    public function isType(string $type) : bool
    {
        if (!isset(self::$ALIAS[$type])) {
            throw new \InvalidArgumentException('type is invalid given ' . $type);
        }
        return $this->type === $type || \in_array($this->type, self::$ALIAS[$type]);
    }

    /**
     * @return string
     */
    public function getTypeName() : string
    {
        if ($this->isStaging()) {
            return self::STAGING;
        }
        if ($this->isProd()) {
            return self::PROD;
        }
        if ($this->isTesting()) {
            return self::TEST;
        }
        if ($this->isDev()) {
            return self::DEV;
        }
        return self::$DEFAULT;
    }

    /**
     * @return bool
     */
    public function isDebug() : bool
    {
        return $this->isDev() || $this->isTesting() || $this->isStaging();
    }

    /**
     * @return bool
     */
    public function isDev() : bool
    {
        return $this->isType(self::DEV);
    }

    /**
     * @return bool
     */
    public function isTesting() : bool
    {
        return $this->isType(self::TEST);
    }

    /**
     * @return bool
     */
    public function isProd() : bool
    {
        return $this->isType(self::PROD);
    }

    /**
     * @return bool
     */
    public function isStaging() : bool
    {
        return $this->isType(self::STAGING);
    }

    /**
     * @return string
     */
    public function getType() : string
    {
        return $this->type;
    }
}
