<?php

namespace Cms\Bundle\Member;

use Cms\Bundle\Member\Exception\ApiFailedException;
use Cms\Bundle\Member\Exception\RuntimeException;
use Cms\Client\Member\Model\MemberRestfulModel;
use Dflydev\FigCookies\FigResponseCookies;
use Dflydev\FigCookies\SetCookie;
use League\OAuth2\Client\Provider\AbstractProvider;
use League\Route\Http\Exception\UnauthorizedException;
use Move\Http\RequestUtils;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Zend\Diactoros\Response\RedirectResponse;

/**
 * Class AbstractLoginProviderController
 * @package Cms\Bundle\Member
 */
abstract class AbstractLoginProviderController implements
    LoggerAwareInterface
{
    use LoggerAwareTrait;

    /**
     * @var MemberLoginService
     */
    private $memberLoginService;

    /**
     * AbstractLoginProviderController constructor.
     * @param MemberLoginService $memberLoginService
     */
    public function __construct(MemberLoginService $memberLoginService)
    {
        $this->memberLoginService = $memberLoginService;
    }

    /**
     * @return \Cms\Bundle\Member\MemberLoginService
     */
    public function getLoginService() : MemberLoginService
    {
        return $this->memberLoginService;
    }

    /**
     * Fourni le provider de login de l'instance
     * @return \League\OAuth2\Client\Provider\AbstractProvider
     */
    abstract protected function getProvider() : AbstractProvider;

    /**
     * Permet de récupérer le retour effectué par le provider de login.
     * $member est non null si le member existe deja suite au login
     * la response doit être modifiee en conséquence
     * @param \Psr\Http\Message\ResponseInterface $response
     * @param string $email
     * @param \Cms\Client\Member\Model\MemberRestfulModel|null $member
     * @return ResponseInterface
     */
    abstract protected function handleCallback(
        ResponseInterface $response,
        string $email,
        MemberRestfulModel $member = null
    ) : ResponseInterface;

    /**
     * @param ServerRequestInterface $request
     * @return ResponseInterface|RedirectResponse
     * @throws \RuntimeException
     * @throws \Psr\Cache\InvalidArgumentException
     * @throws \League\Flysystem\FileNotFoundException
     * @throws \InvalidArgumentException
     */
    public function login(ServerRequestInterface $request)
    {
        // récupération du provider
        $provider = $this->getProvider();
        // création de l'url pour autorisation
        $authUrl = $provider->getAuthorizationUrl();

        // redir vers le provider
        $providerResponse = new RedirectResponse($authUrl, 302);
        $providerResponse = $providerResponse->withHeader('Cache-Control', 'private, no-cache, no-store');

        // enregistrement du state dans un cookie
        $datasForCookie = ['state' => $provider->getState()];
        $cookie = SetCookie::create(MemberHelper::COOKIE_STATEAUTH);
        $cookie = $cookie
            ->withHttpOnly(true)
            ->withDomain($request->getUri()->getHost())
            ->withPath('/')
            ->withValue(json_encode($datasForCookie));
        $providerResponse = FigResponseCookies::set($providerResponse, $cookie);

        // on vide le cookie au cas où il est déjà connecté
        $cookie = RequestUtils::GetCookieValue($request, MemberHelper::COOKIE_MBRLOGIN);
        if ($cookie) {
            $providerResponse = MemberHelper::removeCookie(
                $providerResponse,
                MemberHelper::COOKIE_MBRLOGIN
            );
        }

        return $providerResponse;
    }

    /**
     * @param ServerRequestInterface $request
     * @param ResponseInterface $response
     * @return ResponseInterface|\Zend\Diactoros\Response\JsonResponse
     * @throws \InvalidArgumentException
     * @throws \Cms\Bundle\Member\Exception\ApiFailedException
     * @throws \Cms\Bundle\Member\Exception\RuntimeException
     * @throws \League\Route\Http\Exception\UnauthorizedException
     * @throws UnauthorizedException
     */
    public function callback(ServerRequestInterface $request, ResponseInterface $response)
    {
        //recuperation des queryParams
        $queryParams = $request->getQueryParams();

        // recuperation du cookie pour test du state
        $cookie = RequestUtils::GetCookieValue($request, MemberHelper::COOKIE_STATEAUTH);
        $cookieDatas = json_decode($cookie);
        if (empty($queryParams['state'] || ($queryParams['state'] !== $cookieDatas['state']))) {
            throw new UnauthorizedException('Invalid state');
        }

        // récupération du provider
        $provider = $this->getProvider();
        // récupération du token
        $token = $provider->getAccessToken('authorization_code', [
            'code' => RequestUtils::getQueryValue($request, 'code'),
        ]);

        // récup info membre
        $user = $provider->getResourceOwner($token);
        if (method_exists($user, 'getEmail')) {
            $email = $user->getEmail();
        } else {
            throw new RuntimeException('email manquant');
        }

        // check si un user correspond à l'email
        // vérification d'unicité
        try {
            $memberModel = $this->getLoginService()->getMemberClient()
                ->getItemByEmail($email, true);
        } catch (\Exception $e) {
            throw new ApiFailedException(
                'cannot retrieve member info for email ' . $email,
                $e,
                MemberErrorCode::FETCH_MEMBER_API_FAILED
            );
        }

        // recuperation de la response du callback
        $response = $this->handleCallback($response, $email, $memberModel);
        $response = $response->withHeader('Cache-Control', 'private, no-cache, no-store');
        // suppression du cookie
        $response = MemberHelper::removeCookie($response, MemberHelper::COOKIE_STATEAUTH);

        return $response;
    }
}
