<?php

namespace Cms\LinkResource;

use Move\ObjectMapper\ObjectTransformer;

/**
 * Class FlattenResourcesTrait
 * @package Cms\Client
 * @property array $resources
 */
trait FlattenResourcesTrait
{

    /** @var  array */
    protected $resources;

    /**
     * @param LinkResourceModel|array $resourceData
     * @return LinkResourceModel
     */
    abstract public function handleResource($resourceData);

    /**
     * @param \Cms\LinkResource\LinkResourceType|string $resourceType
     * @param \Cms\LinkResource\LinkResourcePrefsIdentifier|string $identifier
     * @return LinkResourceModel[]
     */
    public function getResourceByPrefIdentifier($resourceType, $identifier) : array
    {
        $identifier = (string)$identifier;
        if (!LinkResourcePrefsIdentifier::isValid($identifier)) {
            throw new \InvalidArgumentException('cannot retrieve by undefined identifier : ' . $identifier);
        }
        $resources = $this->getResourceByType($resourceType, 'identifier', $identifier);
        return $resources;
    }

    /**
     * @param LinkResourceType|string $resourceType
     * @param null|string $prefName
     * @param null|string $prefValue
     * @return LinkResourceModel[]
     * @throws \InvalidArgumentException
     */
    public function getResourceByType($resourceType, $prefName = null, $prefValue = null)
    {
        $resourceType = (string)$resourceType;
        if (!LinkResourceType::isValid($resourceType)) {
            throw new \InvalidArgumentException('cannot retrieve by undefined type : ' . $resourceType);
        }
        if (empty($this->resources[$resourceType])) {
            return [];
        }
        $resources = array_map([$this, 'convertResource'], $this->resources[$resourceType]);

        // filtre sur les prefs
        if ($prefName) {
            foreach ($resources as &$resource) {
                if ($resource && $resource->link_prefs instanceof LinkResourcePrefsModel) {
                    if (!isset($resource->link_prefs[$prefName])
                        || $resource->link_prefs[$prefName] != $prefValue
                    ) {
                        $resource = null;
                    }
                }
            }
            unset($resource);
        }
        return array_values(array_filter($resources));
    }

    /**
     * @param \Cms\LinkResource\LinkResourceModel $resourceModel
     * @return $this
     * @throws \UnexpectedValueException
     */
    public function removeInResourceTree(LinkResourceModel $resourceModel)
    {
        $resources = $this->getFlattenResources();
        $this->resources = [];
        foreach ($resources as $resource) {
            if (!$this->isEqualResource($resourceModel, $resource)) {
                $this->addInResourceTree($resource);
            }
        }
        return $this;
    }

    /**
     * @param LinkResourceType|string $resourceType
     * @param string $resourceRef
     * @param string $resourceProvider
     * @param array $otherProps
     * @return LinkResourceModel
     * @throws \UnexpectedValueException
     */
    public function addResourceType(
        $resourceType,
        $resourceRef,
        $resourceProvider,
        array $otherProps = []
    ) {
        $resource = $this->convertResource(array_merge($otherProps, [
            'link_type' => $resourceType,
            'link_ref' => $resourceRef,
            'link_provider' => $resourceProvider,
        ]));
        $this->addInResourceTree($resource);
        return $resource;
    }

    /**
     * @param \Cms\LinkResource\LinkResourceModel $resource
     * @param \Cms\LinkResource\LinkResourceModel $resourceModel
     * @return bool
     */
    private function isEqualResource(
        LinkResourceModel $resource,
        LinkResourceModel $resourceModel
    ) : bool {
        if (!empty($resourceModel->id)
            && !empty($resource->id)
            && (int)$resourceModel->id === (int)$resource->id
        ) {
            return true;
        }
        if ((string)$resource->link_ref === (string)$resourceModel->link_ref
            && (string)$resource->link_provider === (string)$resourceModel->link_provider
            && (string)$resource->link_type === (string)$resourceModel->link_type
        ) {
            return true;
        }
        return false;
    }

    /**
     * @param LinkResourceModel $resourceModel
     * @return bool
     * @throws \UnexpectedValueException
     */
    public function hasResource(LinkResourceModel $resourceModel) : bool
    {
        foreach ($this->getFlattenResources() as $resource) {
            if ($this->isEqualResource($resource, $resourceModel)) {
                return true;
            }
        }
        return false;
    }

    /**
     * @param LinkResourceModel $model
     * @return $this
     * @throws \UnexpectedValueException
     */
    final public function addInResourceTree(LinkResourceModel $model)
    {
        if (false !== $this->hasResource($model)) {
            return $this;
        }
        $resourceType = (string)$model->link_type;
        if (empty($this->resources[$resourceType]) || !\is_array($this->resources[$resourceType])) {
            $this->resources[$resourceType] = [];
        }
        $this->resources[$resourceType][] = $model;
        return $this;
    }

    /**
     * @return AbstractLinkResourceModel[]
     * @throws \UnexpectedValueException
     */
    public function getFlattenResources()
    {
        $value = !empty($this->resources)
            ? \call_user_func_array('array_merge', $this->resources)
            : [];
        foreach ($value as &$resource) {
            if (!empty($resource)) {
                $resource = $this->convertResource($resource);
            } else {
                $resource = null;
            }
        }
        return array_values(array_filter($value));
    }

    /**
     * @deprecated
     * @return array
     */
    public function getArrayResources()
    {
        return $this->getResourceTree();
    }

    /**
     * @return array
     */
    public function getResourceTree() : array
    {
        return array_map(function ($resourceList) {
            if (!\is_array($resourceList)) {
                return [];
            }
            return array_map(function ($resource) {
                if ($resource instanceof LinkResourceModel) {
                    return (new ObjectTransformer())->transform($resource);
                }
                return \is_array($resource) ? $resource : [];
            }, $resourceList);
        }, $this->resources ?: []);
    }

    /**
     * @param AbstractLinkResourceModel|array $resourceData
     * @return AbstractLinkResourceModel
     * @throws \UnexpectedValueException
     */
    private function convertResource($resourceData)
    {
        if ($resourceData instanceof LinkResourceModel) {
            return $resourceData;
        }
        $resource = $this->handleResource($resourceData);
        if (!$resource instanceof LinkResourceModel) {
            throw new \UnexpectedValueException('link resource incorrectly handled');
        }
        return $resource;
    }
}
