<?php


namespace Cms\Model;

use POM\SerializableSqlInterface;

/**
 * Class AbstractTypeSet
 * @package Cms\Model
 */
abstract class AbstractTypeSet implements
    \IteratorAggregate,
    \ArrayAccess,
    SerializableSqlInterface,
    \Countable
{

    /** @var \ArrayObject */
    private $set;

    /**
     * FeatureFieldFlagSet constructor.
     * @param array|string|null $initialSet
     */
    public function __construct($initialSet = null)
    {
        if (\is_string($initialSet)) {
            $initialSet = array_map('trim', explode(',', $initialSet));
        }
        if (!\is_array($initialSet)) {
            $initialSet = null;
        }
        $this->set = new \ArrayObject($initialSet ?: []);
    }

    /**
     * @param string $value
     * @return bool
     */
    abstract protected function validateValue(string $value) : bool;

    /**
     * @return \ArrayIterator
     */
    public function getIterator() : \ArrayIterator
    {
        return $this->set->getIterator();
    }

    /**
     * @return int
     */
    public function count() : int
    {
        return $this->set->count();
    }

    /**
     * @return array
     */
    public function getArrayCopy() : array
    {
        return $this->set->getArrayCopy();
    }

    /**
     * @param string $flag
     * @return bool
     */
    public function hasFlag(string $flag) : bool
    {
        return \in_array($flag, $this->getArrayCopy());
    }

    /**
     * @param string $flag
     * @throws \InvalidArgumentException
     */
    public function pushFlag(string $flag)
    {
        if (!$this->validateValue($flag)) {
            throw new \InvalidArgumentException('the flag ' . $flag . ' is invalid');
        }
        $this->set->append($flag);
    }

    /**
     * @return string|null
     */
    public function sqlSerialize()
    {
        if ($this->set->count() === 0) {
            return null;
        }
        return implode(',', $this->set->getArrayCopy());
    }

    /**
     * @param string $serialized
     * @return void
     */
    public function sqlUnserialize($serialized)
    {
        $initSet = [];
        if (\is_string($serialized)) {
            $initSet = array_map('trim', explode(',', $serialized));
            $initSet = array_filter($initSet, [$this, 'validateValue']);
        }
        if (\is_array($serialized)) {
            $initSet = $serialized;
        }
        $this->set = new \ArrayObject($initSet);
    }

    /**
     * Whether a offset exists
     * @link http://php.net/manual/en/arrayaccess.offsetexists.php
     * @param mixed $offset <p>
     * An offset to check for.
     * </p>
     * @return boolean true on success or false on failure.
     * </p>
     * <p>
     * The return value will be casted to boolean if non-boolean was returned.
     * @since 5.0.0
     */
    public function offsetExists($offset) : bool
    {
        return $this->set->offsetExists($offset);
    }

    /**
     * Offset to retrieve
     * @link http://php.net/manual/en/arrayaccess.offsetget.php
     * @param mixed $offset <p>
     * The offset to retrieve.
     * </p>
     * @return mixed Can return all value types.
     * @since 5.0.0
     */
    public function offsetGet($offset)
    {
        return $this->set->offsetGet($offset);
    }

    /**
     * Offset to set
     * @link http://php.net/manual/en/arrayaccess.offsetset.php
     * @param mixed $offset <p>
     * The offset to assign the value to.
     * </p>
     * @param mixed $value <p>
     * The value to set.
     * </p>
     * @return void
     * @throws \InvalidArgumentException
     * @since 5.0.0
     */
    public function offsetSet($offset, $value)
    {
        if (!$this->validateValue($value)) {
            throw new \InvalidArgumentException('the flag ' . $value . ' is invalid');
        }
        $this->set->offsetSet($offset, $value);
    }

    /**
     * Offset to unset
     * @link http://php.net/manual/en/arrayaccess.offsetunset.php
     * @param mixed $offset <p>
     * The offset to unset.
     * </p>
     * @return void
     * @since 5.0.0
     */
    public function offsetUnset($offset)
    {
        $this->set->offsetUnset($offset);
    }
}
