<?php


namespace Cms\Search;

use Cms\Search\Provider\Elastica\ElasticSearchResultSet;

/**
 * Class SearchStackPaginator
 * @package Cms\Search
 */
class SearchStackPaginator
{

    /** @var int */
    private $byPage;

    /** @var int */
    private $currentPage;

    /** @var SearchClientSeekableInterface */
    private $seekableClient;

    /**
     * SearchStackPaginator constructor.
     * @param SearchClientSeekableInterface $seekableClient
     * @param int $byPage
     * @param int $currentPage
     */
    public function __construct(
        SearchClientSeekableInterface $seekableClient,
        int $byPage,
        int $currentPage
    ) {
        $this->seekableClient = $seekableClient;
        $this->byPage = $byPage;
        $this->currentPage = $currentPage;
    }

    /**
     * @param array $searchStack
     * @return array
     * @throws \UnexpectedValueException
     * @throws \InvalidArgumentException
     */
    public function __invoke(array $searchStack)
    {
        return $this->executeStack($searchStack);
    }

    /**
     * @param callable[] $searchStack
     * @param array $results
     * @param int $searchTotalHits
     * @return \Cms\Search\Provider\Elastica\ElasticSearchResultElement[]
     * @throws \UnexpectedValueException
     * @throws \InvalidArgumentException
     */
    public function executeStack(
        array $searchStack,
        array &$results = [],
        int &$searchTotalHits = 0
    ) {
        $resultsCount = count($results);

        // init vars
        $byPage = $this->byPage;
        $pageNum = $this->currentPage;

        // on calcul la page a remonté
        $subLimit = $byPage - $resultsCount;
        $pgoffset = ($pageNum * $byPage) - $searchTotalHits - $byPage + $resultsCount % $byPage;
        $client = clone $this->seekableClient;
        $client
            ->setLimit($subLimit)
            ->setOffset($pgoffset);

        // execution de la recherche
        $executor = array_shift($searchStack);
        if (!\is_callable($executor)) {
            throw new \InvalidArgumentException('every element in the execute stack must be callable');
        }
        $subResults = $executor($client);
        if (!$subResults instanceof ElasticSearchResultSet) {
            throw new \UnexpectedValueException('the executor must return a valid ElasticSearchResultSet');
        }
        $searchTotalHits += $subResults->getTotalHits();
        $results = array_merge($results, iterator_to_array($subResults, false));

        //var_dump(count($searchStack), $pgoffset, $resultsCount,
        // $searchTotalHits, $subResults->getTotalHits(), count($results), '--');

        if ($searchStack && count($results) < $byPage) {
            $this->executeStack($searchStack, $results, $searchTotalHits);
        }
        return $results;
    }
}
