<?php


namespace Move\Search;

use Elastica\Client;
use Elastica\Index;

/**
 * Class AbstractIndexFactory
 * @package Move\Search
 */
abstract class AbstractIndexFactory
{

    /** @var \Elastica\Client */
    private $elasticaClient;

    /** @var \Move\Search\ElasticMappingBuilderInterface[] */
    private $mappingBuilders;

    /**
     * CmsIndexFactory constructor.
     * @param \Elastica\Client $elasticaClient
     * @param \Move\Search\ElasticMappingBuilderInterface[] $mappingBuilders
     * @throws \InvalidArgumentException
     */
    public function __construct(
        Client $elasticaClient,
        array $mappingBuilders = []
    ) {
        $this->elasticaClient = $elasticaClient;
        foreach ($mappingBuilders as $builder) {
            if (!$builder instanceof ElasticMappingBuilderInterface) {
                throw new \InvalidArgumentException(
                    'mapping builder is not an instance of ElasticMappingBuilderInterface'
                );
            }
        }
        $this->mappingBuilders = $mappingBuilders;
    }

    /**
     * @param bool $recreate
     * @param int|null $shards
     * @return \Move\Search\ElasticIndexBuilderInterface
     */
    abstract public static function createIndexBuilder(
        bool $recreate = false,
        int $shards = null
    ) : ElasticIndexBuilderInterface;

    /**
     * @return string
     */
    abstract public static function getIndexName() : string;

    /**
     * @param bool $recreate
     * @param int|null $shards
     * @return \Elastica\Index
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     * @throws \Elastica\Exception\ResponseException
     * @throws \Elastica\Exception\InvalidException
     */
    public function createIndex(bool $recreate = false, int $shards = null) : Index
    {
        $indexBuilder = static::createIndexBuilder($recreate, $shards);
        $index = $indexBuilder->createIndex($this->elasticaClient, null, $this->mappingBuilders);
        return $index;
    }

    /**
     * @param string|null $rotateIndex
     * @param bool $recreate
     * @param int|null $shards
     * @return \Elastica\Index
     */
    public function rotateIndex(string $rotateIndex = null, bool $recreate = true, int $shards = null) : Index
    {
        $indexBuilder = static::createIndexBuilder($recreate, $shards);
        $indexBuilder->setIndexName(static::getIndexName() . '_' . ($rotateIndex ?: date('Ymdhi')));
        $index = $indexBuilder->createIndex($this->elasticaClient, null, $this->mappingBuilders);
        return $index;
    }

    /**
     * @param bool $recreate
     * @param int|null $shards
     * @return \Elastica\Index
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     * @throws \Elastica\Exception\ResponseException
     * @throws \Elastica\Exception\InvalidException
     */
    public function createIndexNoAliases(bool $recreate = true, int $shards = null) : Index
    {
        $indexBuilder = static::createIndexBuilder($recreate, $shards);
        $indexBuilder->setAliases([]);
        $index = $indexBuilder->createIndex($this->elasticaClient, null, $this->mappingBuilders);
        return $index;
    }

    /**
     * @param \Elastica\Index $index
     * @return \Elastica\Index
     */
    public function finalizeIndexRotation(Index $index) : Index
    {
        $indexBuilder = static::createIndexBuilder();
        return $indexBuilder->createAliases($index, [static::getIndexName()]);
    }

    /**
     * @param string|null $indexName
     * @param array $aliases
     * @return \Elastica\Index
     * @throws \RuntimeException
     */
    public function createAliases(string $indexName = null, array $aliases = []) : Index
    {
        $index = $this->getIndex($indexName);
        return static::createIndexBuilder()->createAliases($index, $aliases);
    }

    /**
     * @param string|null $indexName
     * @return bool
     * @throws \UnexpectedValueException
     */
    public function indexExist(string $indexName = null) : bool
    {
        $index = $this->elasticaClient->getIndex($indexName ?: static::getIndexName());
        return $index->exists();
    }

    /**
     * @param string|null $indexName
     * @return \Elastica\Index
     * @throws \UnexpectedValueException
     */
    public function getIndex(string $indexName = null) : Index
    {
        $index = $this->elasticaClient->getIndex($indexName ?: static::getIndexName());
        // pas de creation induite
        if (!$index->exists()) {
            throw new \UnexpectedValueException('index does not exist');
        }
        return $index;
    }

    /**
     * @return \Elastica\Type\Mapping[]
     * @throws \UnexpectedValueException
     */
    public function createMappings() : array
    {
        $index = $this->getIndex();
        $mappings = [];
        foreach ($this->getMappings() as $mappingBuilder) {
            $mappings[] = $mappingBuilder->createMapping($index);
        }
        return $mappings;
    }

    /**
     * @return \Move\Search\ElasticMappingBuilderInterface[]
     */
    protected function getMappings() : array
    {
        return $this->mappingBuilders;
    }
}
