<?php


namespace Cms\Bundle\Member;

use League\OAuth2\Client\Provider\AbstractProvider;
use League\OAuth2\Client\Provider\Google;
use Move\Config;
use Move\Environment;
use Zend\Diactoros\Uri;

/**
 * Class AbstractGoogleProviderController
 * @package Cms\Bundle\Member
 */
abstract class AbstractGoogleProviderController extends AbstractLoginProviderController
{
    /** @var \Move\Config */
    private $config;

    /**
     * AbstractGoogleProviderController constructor.
     * @param MemberLoginService $memberLoginService
     * @param \Move\Config $config
     * @throws \InvalidArgumentException
     */
    public function __construct(MemberLoginService $memberLoginService, Config $config)
    {
        parent::__construct($memberLoginService);
        $this->config = $config;
        if (!isset(
            $this->config['oauth.google.clientid'],
            $this->config['oauth.google.secret']
        )
        ) {
            throw new \InvalidArgumentException('oauth.google.[clientid,secret] must be set in config');
        }
    }

    /**
     * @return \League\OAuth2\Client\Provider\AbstractProvider
     * @throws \InvalidArgumentException
     */
    protected function getProvider() : AbstractProvider
    {
        $redirectUri = $this->getRedirectUri();

        // changement du host en local pour restriction de host google
        $redirectUri = new Uri($redirectUri);
        if ((new Environment())->isDev()) {
            $redirectUri = $redirectUri->withHost('localhost');
        }

        $provider = new Google([
            'clientId' => $this->config['oauth.google.clientid'],
            'clientSecret' => $this->config['oauth.google.secret'],
            'redirectUri' => (string)$redirectUri,
            'hostedDomain' => $redirectUri->getScheme() . '://' . $redirectUri->getHost(),
        ]);
        return $provider;
    }

    /**
     * @return string
     */
    abstract protected function getRedirectUri() : string;
}
