<?php


namespace Cms\Search\Campsite;

use Cms\Client\Geodata\Model\AdminRestfulModel;
use Cms\Client\Geodata\Model\DivisionRestfulModel;
use Cms\Model\Campsite\CampsiteAgendaType;
use Cms\Model\Campsite\CampsiteModel;
use Cms\Model\Campsite\CampsiteState;
use Cms\Model\Campsite\Feature\FeatureCode;
use Cms\Model\Campsite\Guide\GuideModel;
use Cms\Model\Campsite\Product\ProductModel;
use Cms\Search\Mapping\AddressDocumentTrait;
use Cms\Search\Mapping\AddressMapping;
use Cms\Search\Mapping\AddressNestedMapping;
use Cms\Search\Mapping\GeoAdminDocumentTrait;
use Cms\Search\Mapping\GeoAdminNestedMapping;
use Elastica\Document;
use Move\Search\Analyzer\FrenchSearchAnalyzer;
use Move\Search\Analyzer\FrenchTextAnalyzer;
use Move\Search\Analyzer\GeoFrenchAnalyzer;
use Move\Search\Analyzer\GeoFrenchSearchAnalyzer;
use Move\Search\ElasticMappingBuilder;
use Move\Search\Mapping\ElasticMappingProperty;
use Move\Search\Mapping\KeywordMapping;
use Move\Search\Mapping\ObjectMapping;
use Move\Search\Mapping\TextMapping;
use Move\Utils\Str;

/**
 * Class CampsiteIndexFactory
 * @package Cms\Search\Campsite
 */
class CampsiteIndexFactory
{
    use AddressDocumentTrait;
    use GeoAdminDocumentTrait;

    /** Nom du type pour les campsite dans l'index */
    public const MAPPING_NAME = 'cms_campsite';

    /**
     * @param bool $nested
     * @return $this|\Move\Search\Mapping\ObjectMapping
     */
    public static function createFeatureMapping(bool $nested = false)
    {
        $features = [
            new KeywordMapping('feature_code'),
            new KeywordMapping('value', false),
            new ElasticMappingProperty('distance', 'float', false),
            new KeywordMapping('distance_type'),
            new ElasticMappingProperty('quantity', 'long'),
            new ElasticMappingProperty('is_free', 'boolean'),
        ];

        if (!$nested) {
            return new ObjectMapping('features', $features);
        }
        return (new ObjectMapping('features_nested', $features))->isNested(true);
    }

    /**
     * @param bool $nested
     * @return $this|\Move\Search\Mapping\ObjectMapping
     */
    public static function createProductMapping(bool $nested = false)
    {
        $products = [
            new ElasticMappingProperty('id', 'long'),
            new KeywordMapping('value', false),
            new ElasticMappingProperty('quantity', 'long'),
            new ElasticMappingProperty('is_free', 'boolean'),
        ];

        if (!$nested) {
            return new ObjectMapping('campsite_products', $products);
        }
        return (new ObjectMapping('campsite_products_nested', $products))->isNested(true);
    }

    /**
     * @return \Move\Search\ElasticMappingBuilder
     */
    public static function createMappingBuilder() : ElasticMappingBuilder
    {
        $builder = new ElasticMappingBuilder(self::MAPPING_NAME, [
            new ElasticMappingProperty('id', 'long'),
            new ElasticMappingProperty('scope_id', 'long'),

            new KeywordMapping('state'),
            new KeywordMapping('type'),
            new KeywordMapping('language'),

            new ElasticMappingProperty('campsite_opened_at', 'date', false),
            new ElasticMappingProperty('campsite_closed_at', 'date', false),
            new ElasticMappingProperty('campsite_always_open', 'boolean'),
            new ElasticMappingProperty('campsite_accept_vac', 'boolean'),

            new ElasticMappingProperty('classmnt_nbstars', 'integer', false),

            new KeywordMapping('slug'),

            new KeywordMapping('campsite_acces'),

            (new TextMapping('campsite_name'))->withAnalyzer(
                new FrenchTextAnalyzer(),
                new FrenchSearchAnalyzer()
            ),

            (new TextMapping('campsite_description', false))->withAnalyzer(
                new FrenchTextAnalyzer(),
                new FrenchSearchAnalyzer()
            ),

            new GeoAdminNestedMapping(),

            new ObjectMapping('geo_divisions', [
                new ElasticMappingProperty('id', 'long'),
                new KeywordMapping('division_slug'),
                (new TextMapping('division_name'))->withAnalyzer(
                    new GeoFrenchAnalyzer(),
                    new GeoFrenchSearchAnalyzer()
                ),
                (new ObjectMapping('division_name_i18n', [ // autre langue
                    new ElasticMappingProperty('id', 'long'),
                    new KeywordMapping('language'),
                    new KeywordMapping('division_slug'),
                    (new TextMapping('division_name'))->withAnalyzer(
                        new GeoFrenchAnalyzer(),
                        new GeoFrenchSearchAnalyzer()
                    ),
                ], false))->isNested(true),
            ], false),

            new AddressMapping(),
            new AddressNestedMapping(),

            new ElasticMappingProperty('campsite_groups', 'long', false),
            new ElasticMappingProperty('campsite_thematics', 'long', false),
            new KeywordMapping('campsite_guide_codes'),

            static::createFeatureMapping(),
            static::createFeatureMapping(true),

            new ElasticMappingProperty('campsite_nb_rent', 'long'),

            static::createProductMapping(),
            static::createProductMapping(true),

            new ElasticMappingProperty('created_at', 'date'),
            new ElasticMappingProperty('modified_at', 'date'),

            new ObjectMapping('cms_users', [
                new ElasticMappingProperty('id', 'long'),
                new ElasticMappingProperty('show_name', 'string'),
                new ElasticMappingProperty('email', 'string'),
                new ElasticMappingProperty('state', 'string'),
            ]),
        ]);
        return $builder;
    }

    /**
     * @param \Cms\Model\Campsite\CampsiteModel $campsiteModel
     * @param \Cms\Model\AddressBook\Address\AddressModel[] $addresses
     * @param DivisionRestfulModel[] $divisions
     * @param \Cms\Model\Campsite\CampsiteFeatureModel[] $features
     * @param \Cms\Model\Campsite\CampsiteProductModel[]|\Cms\Model\Campsite\CampsitePackageProductModel[] $products
     * @param int[]|\Cms\Model\Campsite\Thematic\ThematicModel[] $thematics
     * @param int[]|\Cms\Model\Campsite\CampsiteGroupModel[] $groupIds
     * @param AdminRestfulModel[] $admins
     * @param \Cms\Model\Campsite\CampsiteAgendaModel[] $agendas
     * @param array $productRentIds
     * @param \Cms\Model\Campsite\Guide\GuideModel[] $guides
     * @param array $cmsUsers
     * @param array $subscribedThematicIds
     * @return \Elastica\Document
     */
    public static function createDocument(
        CampsiteModel $campsiteModel,
        array $addresses = [],
        array $divisions = [],
        array $features = [],
        array $products = [],
        array $thematics = [],
        array $groupIds = [],
        array $admins = [],
        array $agendas = [],
        array $productRentIds = [],
        array $guides = [],
        array $cmsUsers = [],
        array $subscribedThematicIds = []
    ) : Document {
        $documentData = [
            'id' => $campsiteModel->id,
            'scope_id' => $campsiteModel->scope_id,
            'state' => (string)$campsiteModel->state,
            'type' => (string)$campsiteModel->type,
            'language' => (string)$campsiteModel->language,
            'slug' => $campsiteModel->slug,
            'classmnt_nbstars' => 0,
            'created_at' => $campsiteModel->created_at->format(DATE_W3C),
            'modified_at' => $campsiteModel->modified_at->format(DATE_W3C),
            'campsite_opened_at' => null,
            'campsite_closed_at' => null,
            'campsite_always_open' => false,
            'campsite_name' => $campsiteModel->name,
            'campsite_acces' => $campsiteModel->acces,
            'campsite_description' => $campsiteModel->description ?: null,
            'campsite_groups' => array_map(function ($group) {
                return is_numeric($group) ? $group : $group->group_id ?? null;
            }, $groupIds) ?: null,
            'campsite_thematics' => array_map(function ($thematic) {
                return is_numeric($thematic) ? $thematic : $thematic->id ?? null;
            }, $thematics) ?: null,
            'subscribed_thematics' => $subscribedThematicIds ?? [],
            'geo_admins_nested' => array_map(
                [self::class, 'buildGeoAdminMapping'],
                $admins
            ),
            'geo_divisions' => array_map(
                [self::class, 'transformDivision'],
                $divisions
            ) ?: null,
            'campsite_accept_vac' => true,
            'addresses' => [],
            'addresses_nested' => [],
            'features' => [],
            'features_nested' => [],
            'campsite_nb_rent' => 0,
            'campsite_guide_codes' => [],
            'campsite_products' => [],
            'campsite_products_nested' => [],
            'cms_users' => [],
        ];

        foreach ($guides as $guideModel) {
            if ($guideModel instanceof GuideModel) {
                $documentData['campsite_guide_codes'][] = $guideModel->code;
            }
        }

        // check date ouverture/fermeture
        $openDate = $closeDate = null;
        $curYear = date_create()->format('Y');
        foreach ($agendas as $agenda) {
            if (!$agenda->close_date || !$agenda->open_date || $agenda->type === CampsiteAgendaType::SERVICE) {
                continue;
            }
            if (!$openDate instanceof \DateTime
                || $openDate->format('z') > $agenda->open_date->format('z')
            ) {
                $openDate = date_create_from_format(
                    'dmY',
                    $agenda->open_date->format('dm') . $curYear
                );
            }
            if (!$closeDate instanceof \DateTime
                || $closeDate->format('z') < $agenda->close_date->format('z')
            ) {
                $closeDate = date_create_from_format(
                    'dmY',
                    $agenda->close_date->format('dm') . $curYear
                );
            }
        }
        if ($openDate && $closeDate) {
            $documentData['campsite_opened_at'] = $openDate->format(DATE_W3C);
            $documentData['campsite_closed_at'] = $closeDate->format(DATE_W3C);
            if ($openDate->format('dm') === '0101' && $closeDate->format('dm') === '3112') {
                $documentData['campsite_always_open'] = true;
            }
        }

        // ajout des cms users
        foreach ($cmsUsers as $cmsUser) {
            $documentData['cms_users'][] = [
                'id' => $cmsUser->id,
                'show_name' => $cmsUser->show_name,
                'email' => $cmsUser->email,
                'state' => $cmsUser->state->getValue(),
            ];
        }

        // ajout des features
        foreach ($features as $feature) {
            if ((string)$feature->feature_code === FeatureCode::FERMETURE) {
                $documentData['state'] = CampsiteState::CLOSE;
                continue;
            }
            if ((string)$feature->feature_code === FeatureCode::PRLNOVAC
                && $feature->value
            ) {
                $documentData['campsite_accept_vac'] = false;
                continue;
            }
            $documentData['features'][] = [
                'feature_code' => $feature->feature_code,
                'value' => $feature->value,
                'distance' => $feature->distance,
                'distance_type' => (string)$feature->distance_type,
                'quantity' => $feature->quantity,
                'is_free' => $feature->is_free,
            ];
            if ((string)$feature->feature_code === FeatureCode::CLASSEMENT) {
                $documentData['classmnt_nbstars'] = substr_count($feature->value, '*') ?: 0;
            }
        }
        $documentData['features_nested'] = $documentData['features'];

        // ajout des products
        foreach ($products as $product) {
            if (!empty($product->quantity)
                && \is_numeric($product->quantity)
                && \in_array($product->product_id, $productRentIds)
            ) {
                $documentData['campsite_nb_rent'] += $product->quantity;
            }
            $documentData['campsite_products'][] = [
                'id' => $product->product_id,
                'value' => $product->value,
                'quantity' => $product->quantity,
                'is_free' => $product instanceof ProductModel ? $product->is_free : false,
            ];
        }
        $documentData['campsite_products_nested'] = $documentData['campsite_products'];

        // ajouts address
        foreach ($addresses as $type => $addressModel) {
            if (!\is_string($type)) {
                continue;
            }
            $documentData['addresses'][] = self::buildAddressMapping($type, $addressModel);
        }
        $documentData['addresses_nested'] = $documentData['addresses'];

        // envoi des données
        return new Document($campsiteModel->id, $documentData, self::MAPPING_NAME);
    }

    /**
     * @param array $division
     * @return array
     */
    private static function transformDivision(array $division) : array
    {
        $defaulti18n = [
            'language' => 'fre',
            'name' => $division['name'],
            'name_slug' => $division['name_slug'],
        ];
        return [
            'id' => $division['id'],
            'division_name' => $division['name'],
            'division_slug' => $division['name_slug'],
            'division_name_i18n' => array_map(function ($i18n) use ($division) {
                return [
                    'id' => $division['id'],
                    'language' => $i18n['language'],
                    'division_name' => $i18n['name'],
                    'division_slug' => $i18n['name_slug'] ?? Str::toAscii($i18n['name']),
                ];
            }, array_merge($division['i18n'] ?: [], [$defaulti18n])),
        ];
    }
}
