<?php

namespace Cms\Worker;

use Cms\Client\Scheduler\SchedulerTaskClient;
use Cms\Client\Service\SchedulerProvider;
use Cms\Model\Scheduler\LinkResource\LinkResourceModel;
use Cms\Worker\Command\Scheduler\SchedulerCommand;
use Cms\Worker\Scheduler\SchedulerCommandHandler;
use Cms\Worker\Scheduler\SchedulerRabbitMQCommandHandler;
use League\Container\Argument\RawArgument;
use League\Container\ContainerInterface;
use League\Tactician\Handler\MethodNameInflector\HandleInflector;
use Monolog\Formatter\JsonFormatter;
use Monolog\Handler\StreamHandler;
use Monolog\Logger;
use Move\Command\CommandBusInterface;
use Move\Command\TacticianBusFactory;
use Move\Utils\Str;
use Psr\Log\LoggerInterface;

/**
 * Class WorkerCommandFactory
 * @package Cms\Worker
 */
class WorkerCommandFactory
{

    /** @var string */
    public static $WORKER_NAMESPACE = __NAMESPACE__ . "\\Command";

    /** @var array */
    protected static $NS_ACTION_TYPE = [];

    /**
     * @param array $actionTypes
     * @param string $namespace
     */
    public static function setNamespaceForType(array $actionTypes, string $namespace)
    {
        foreach ($actionTypes as $actionType) {
            self::$NS_ACTION_TYPE[$actionType] = $namespace;
        }
    }

    /**
     * @param string $actionType
     * @param LinkResourceModel $resource
     * @param array $actionParams
     * @return mixed
     * @throws \ReflectionException
     */
    public static function actionTypeToCommand(
        $actionType,
        LinkResourceModel $resource = null,
        $actionParams = null
    ) {
        // selection du namespace
        $actionType = (string)$actionType;
        $nsName = self::$WORKER_NAMESPACE;
        if (!empty(self::$NS_ACTION_TYPE[$actionType])) {
            $nsName = self::$NS_ACTION_TYPE[$actionType];
        }

        // On récupère le nom de la commande à partir de l'actionType
        $exploded = explode('.', $actionType);
        switch ($exploded[0]) {
            case 'idx':
                $exploded[0] = 'index';
                break;
            case 'art':
                $exploded[0] = 'article';
                break;
        }
        // On construit le nom de la commande avec le namespace
        $commandName = Str::camelize(implode('-', $exploded)) . 'Command';
        $commandFullName = $nsName . "\\" . Str::camelize($exploded[0]) . "\\" . $commandName;

        // On récupère les bons paramètres pour la commande
        $params = [];
        switch ($commandName) {
            default:
                if (\is_string($actionParams)) {
                    $actionParams = @unserialize($actionParams);
                }
                if (\is_array($actionParams)) {
                    $params = $actionParams;
                }
                break;
            case 'ArticlePublishCommand':
                $params = $resource ? [$resource->link_ref] : [null];
                break;
        }

        // On crée la réflection de la commande
        $reflection = new \ReflectionClass($commandFullName);

        // On renvoie la commande instanciée avec les params
        return $reflection->newInstanceArgs($params);
    }

    /**
     * @param ContainerInterface $container
     * @param CommandBusInterface $commandBus
     * @param int|null $loggerLevel
     * @return \Move\Command\CommandBusInterface
     * @throws \InvalidArgumentException
     * @throws \Exception
     */
    public static function createSchedulerBus(
        ContainerInterface $container,
        CommandBusInterface $commandBus,
        $loggerLevel = null
    ) {
        $logStream = self::initBus($container, $loggerLevel);

        // ajoute les command Handler
        $container->add(SchedulerCommandHandler::class)
            ->withArgument(SchedulerTaskClient::class)
            ->withArgument(new RawArgument($commandBus))
            ->withArgument(new RawArgument($logStream));

        // cmd bus
        $bus = TacticianBusFactory::containerMap(
            $container,
            [
                SchedulerCommand::class => SchedulerCommandHandler::class,
            ],
            new HandleInflector()
        );
        return $bus;
    }

    /**
     * @param ContainerInterface $container
     * @param CommandBusInterface $commandBus
     * @param null $loggerLevel
     * @return CommandBusInterface
     * @throws \InvalidArgumentException
     * @throws \Exception
     */
    public static function createRabbitMQBus(
        ContainerInterface $container,
        CommandBusInterface $commandBus,
        $loggerLevel = null
    ) {
        $logStream = self::initBus($container, $loggerLevel);

        // ajoute les command Handler
        $container->add(SchedulerRabbitMQCommandHandler::class)
            ->withArgument(SchedulerTaskClient::class)
            ->withArgument(new RawArgument($commandBus))
            ->withArgument(new RawArgument($logStream));

        // cmd bus
        $bus = TacticianBusFactory::containerMap(
            $container,
            [
                SchedulerCommand::class => SchedulerRabbitMQCommandHandler::class,
            ],
            new HandleInflector()
        );
        return $bus;
    }

    /**
     * @param ContainerInterface $container
     * @param null $loggerLevel
     * @return resource
     * @throws \InvalidArgumentException
     * @throws \Exception
     */
    protected static function initBus(ContainerInterface $container, $loggerLevel = null)
    {
        // ajoute le provider
        $container->addServiceProvider(new SchedulerProvider());

        // ajoute le stream
        $logStream = fopen('php://temp', 'r+');
        $logHandler = new StreamHandler($logStream, $loggerLevel ?: Logger::NOTICE);
        $logHandler->setFormatter(new JsonFormatter());
        $container->inflector(LoggerInterface::class, function ($logger) use ($logHandler) {
            // check pour inflect seulement 1 fois
            if ($logger instanceof Logger
                && !\in_array($logHandler, $logger->getHandlers())
            ) {
                $logger->pushHandler($logHandler);
            }
        });

        return $logStream;
    }
}
