<?php


namespace Move\Image;

/**
 * Class Iptc
 * @package Move\Image
 * @property string $title
 * @property array $keywords
 * @property string $copyright
 * @property string $caption
 * @property string $headline
 * @property string $credit
 * @property string $source
 * @property string $jobtitle
 * @property string $city
 * @property string $state
 * @property string $country
 */
class Iptc
{

    /**
     * Contains the mapping of names to IPTC field numbers
     * @var array
     */
    private static $IPTC_MAPPING = [
        'title' => '2#005',
        'keywords' => '2#025',
        'copyright' => '2#116',
        'caption' => '2#120',
        'headline' => '2#105',
        'credit' => '2#110',
        'source' => '2#115',
        'jobtitle' => '2#085',
        'city' => '2#090',
        'state' => '2#095',
        'country' => '2#101',
    ];

    /** @var  array */
    private $iptc;

    /**
     * @param array $imageinfo
     * @return Iptc
     * @throws \InvalidArgumentException
     */
    public static function createFromInfo(array $imageinfo) : Iptc
    {
        if (!isset($imageinfo['APP13'])) {
            throw new \InvalidArgumentException('iptc field APP13 is empty or invalid');
        }
        // lecture des IPTC
        return static::createFromString($imageinfo['APP13']);
    }

    /**
     * @param string $iptcstring
     * @return \Move\Image\Iptc
     */
    public static function createFromString(string $iptcstring) : Iptc
    {
        // lecture des IPTC
        if (false !== ($iptc = iptcparse($iptcstring))) {
            return static::createFromArray($iptc);
        }
        return new self();
    }

    /**
     * @param array $iptcparsed
     * @return \Move\Image\Iptc
     */
    public static function createFromArray(array $iptcparsed) : Iptc
    {
        $arrData = [];
        foreach (static::$IPTC_MAPPING as $name => $field) {
            if (!isset($iptcparsed[$field])) {
                continue;
            }
            if (count($iptcparsed[$field]) === 1) {
                $arrData[$name] = reset($iptcparsed[$field]);
            } else {
                $arrData[$name] = $iptcparsed[$field];
            }
        }
        $iptc = new self();
        $iptc->iptc = $arrData;
        return $iptc;
    }


    /**
     * @param string $name
     * @return null|string|array
     * @throws \InvalidArgumentException
     */
    public function __get($name)
    {
        if (isset(static::$IPTC_MAPPING[$name])) {
            $result = $this->iptc[$name] ?? null;
            if ($name === 'keywords' && $result !== null && !\is_array($result)) {
                return [$result];
            }
            return $result;
        }
        throw new \InvalidArgumentException($name . ' is not a valid iptc field');
    }

    /**
     * @param string $name
     * @param string|array $value
     * @throws \InvalidArgumentException
     */
    public function __set($name, $value)
    {
        throw new \InvalidArgumentException($name . ' cannot be set');
    }

    /**
     * @param string $name
     * @return bool
     */
    public function __isset($name)
    {
        return isset(static::$IPTC_MAPPING[$name]);
    }

    /**
     * @return array
     */
    public function getArrayCopy() : array
    {
        return $this->iptc;
    }
}
