<?php

namespace Cms\Extension;

use Cms\Cache\CacheHelperTrait;
use Cms\Client\Navigation\Model\NavigationItemRestfulModel;
use Cms\Client\Service\NavigationService;
use Cms\Route\UrlBuilder;
use League\Plates\Engine;
use League\Plates\Extension\ExtensionInterface;
use Move\Cache\CacheAwareInterface;
use Move\Iterator\MapIterator;
use Move\ObjectMapper\ObjectTransformer;
use Psr\Cache\CacheItemInterface;

/**
 * Class MenuExtension
 * @package Cms\Extension
 */
class MenuExtension implements ExtensionInterface, CacheAwareInterface
{
    use CacheHelperTrait;

    /** @var Engine */
    private $engine;

    /** @var NavigationService */
    private $navigationService;

    /** @var int */
    private $cmsScopeId;

    /** @var UrlBuilder */
    private $urlBuilder;

    /** @var int */
    private $cacheTTL;

    /** @var callable|null */
    private $activator;


    /**
     * MenuExtension constructor.
     * @param NavigationService $navigationService
     * @param UrlBuilder $urlBuilder
     * @param int $cmsScopeId
     * @param int $cacheTTL
     */
    public function __construct(
        NavigationService $navigationService,
        UrlBuilder $urlBuilder,
        $cmsScopeId,
        $cacheTTL = 600
    ) {
        $this->navigationService = $navigationService;
        $this->urlBuilder = $urlBuilder;
        $this->cmsScopeId = $cmsScopeId;
        $this->cacheTTL = $cacheTTL;
    }

    /**
     * @param Engine $engine
     */
    public function register(Engine $engine)
    {
        $engine->registerFunction('menu', [$this, 'render']);
        $this->engine = $engine;
    }

    /**
     * @param callable|null $activator
     * @return $this
     */
    public function setActivator(callable $activator = null)
    {
        $this->activator = $activator;
        return $this;
    }

    /**
     * @param string $template
     * @param array $data
     * @return void
     */
    public function render($template, array $data = [])
    {
        $cacheKey = $this->cmsScopeId . '^ExtensionMenu';

        // check dans le cache
        if (!$this->hitFromCache($cacheKey, $this->cacheTTL, $cacheItem)) {
            // récuperation des parents
            $client = $this->navigationService->getItemClient();
            $parents = $client->getParents($this->cmsScopeId);
            // convertion de la liste d'elements
            $data['parents'] = new MapIterator($parents, [$this, 'transformItem']);
            $data['parents'] = iterator_to_array($data['parents']);
            $this->setInCache($cacheItem, $data['parents']);
        } elseif ($cacheItem instanceof CacheItemInterface) {
            $data['parents'] = $cacheItem->get();
        }

        // active l'entrée de menu
        if (!empty($data['parents'])) {
            $requestPath = $data['request_path'] ?? null;
            foreach ($data['parents'] as &$child) {
                $child = $this->activeRecursive($child, $requestPath);
                if ($child['activ']) {
                    break;
                }
            }
            unset($child);
        }

        // ajoute les templates
        if ($this->engine) {
            $menu = $this->engine->make($template);
            echo $menu->render($data);
        }
    }

    /**
     * @param string $str
     * @return string
     */
    protected function stripQuery($str)
    {
        $str = strpos($str, '?') !== false ? explode('?', $str)[0] : $str;
        $str = strpos($str, '#') !== false ? explode('#', $str)[0] : $str;
        return $str;
    }

    /**
     * @param array $result
     * @param string $requestPath
     * @return array
     */
    protected function activeRecursive($result, $requestPath = null)
    {
        $result['activ'] = false;

        // check si l'url a le meme start que celui du menu
        if ($requestPath
            && !empty($result['url'])
            && isset($result['url_loaded']) && $result['url_loaded'] === true
            && (
                (
                    $requestPath === '/'
                    && $requestPath === $this->stripQuery($result['url'])
                )
                || strpos($requestPath, $result['url']) === 0
                || strpos($requestPath, $this->stripQuery($result['url'])) === 0
                || (
                    $this->activator !== null
                    && \call_user_func($this->activator, $result, $requestPath) === true
                )
            )
        ) {
            $result['activ'] = true;
        }

        // parcours les enfant pour activé l'enfant
        if (!empty($result['children'])) {
            $result['children'] = array_map(function ($child) use ($requestPath) {
                return $this->activeRecursive($child, $requestPath);
            }, $result['children']);

            // on remonte en cascade au parent qui est activé
            if (!$result['activ']) {
                foreach ($result['children'] as $child) {
                    if ($child['activ'] === true) {
                        $result['activ'] = true;
                    }
                }
            }
        }

        return $result;
    }

    /**
     * @param NavigationItemRestfulModel $item
     * @param array $result
     * @return bool
     */
    protected function loadUrl(NavigationItemRestfulModel $item, array &$result)
    {
        if (!empty($item->url)) {
            $result['url'] = $item->url;
            return !filter_var($item->url, FILTER_VALIDATE_URL);
        }
        // récuper de la resource pour le link
        if (!empty($item->resources)) {
            $result['url_resource'] = $item->getFlattenResources()[0];
            $result['url'] = $this->urlBuilder->buildUrl($result['url_resource']);
            return true;
        }
        $result['url'] = $this->urlBuilder->buildUrl('default');
        return false;
    }

    /**
     * @param NavigationItemRestfulModel $item
     * @return array
     */
    public function transformItem(NavigationItemRestfulModel $item)
    {
        // modif
        $result = (new ObjectTransformer())->transform($item);

        // default data
        $result['url_resource'] = null;
        $result['url_loaded'] = false;

        // generation des données de l'url
        if ($this->loadUrl($item, $result) === true) {
            $result['url_loaded'] = true;
        }

        // récup des enfant
        $client = $this->navigationService->getItemClient();
        $children = $client->getChildren($item->id);
        if (!empty($children)) {
            $result['children'] = new MapIterator($children, [$this, 'transformItem']);
            $result['children'] = iterator_to_array($result['children']);
        }

        return $result;
    }
}
