<?php


namespace Cms\Model;

use function Move\I18n\countryCodeToLng;
use function Move\I18n\detectLngWithLocale;
use Move\I18n\I18nLanguageCode;
use function Move\I18n\localeCodeToLng;

/**
 * Class I18nHelper
 * @package CampingFrance\Helper
 */
class I18nHelper
{

    /**
     * @param int|null $category
     * @param string|null $fallbackLang
     * @return string|null
     * @deprecated
     */
    public static function detectLngWithLocale(int $category = null, string $fallbackLang = null)
    {
        return detectLngWithLocale($category, $fallbackLang);
    }

    /**
     * @param string $locale
     * @param string|null $fallbackLang
     * @return string
     * @throws \InvalidArgumentException
     * @deprecated
     */
    public static function localeCodeToLng(string $locale, string $fallbackLang = null) : string
    {
        return localeCodeToLng($locale, $fallbackLang);
    }

    /**
     * @param string $countryCode
     * @param string|null $fallbackLang
     * @return string
     * @throws \InvalidArgumentException
     * @deprecated
     */
    public static function countryCodeToLng(
        string $countryCode = null,
        string $fallbackLang = null
    ) : string {
        return countryCodeToLng($countryCode, $fallbackLang);
    }

    /**
     * @param array $i18n
     * @param string $language
     * @param string|null $fallbackLang
     * @return array
     * @throws \InvalidArgumentException
     */
    public static function getTranslateData(
        array $i18n,
        string $language,
        string $fallbackLang = null
    ) : array {
        $language = countryCodeToLng($language);
        foreach ($i18n as $item) {
            if (isset($item['language']) && $item['language'] === $language) {
                if ($item instanceof I18nModelInterface) {
                    return $item->getArrayCopy();
                }
                return (array)$item;
            }
        }
        if ($fallbackLang === null) {
            throw new \InvalidArgumentException('any data for language ' . $language);
        }
        return static::getTranslateData($i18n, $fallbackLang);
    }

    /**
     * @param array $i18n
     * @param string $language
     * @param string $labelField
     * @return string
     * @throws \InvalidArgumentException
     */
    public static function getTranslateLabel(
        array $i18n,
        string $language,
        string $labelField
    ) : string {
        $data = static::getTranslateData($i18n, $language);
        if (isset($data[$labelField])) {
            return (string)$data[$labelField];
        }
        throw new \InvalidArgumentException('label field ' . $labelField . ' does not exist in translate dataset');
    }

    /**
     * @param mixed $object
     * @param string|null $field
     * @param string|null $default
     * @param string|null $lang
     * @return null|string
     * @throws \InvalidArgumentException
     */
    public static function translateObject(
        $object,
        string $field = null,
        string $default = null,
        string $lang = null
    ) {
        // check language
        $lang = $lang ?: detectLngWithLocale();
        if (!I18nLanguageCode::isValid($lang)) {
            throw new \InvalidArgumentException('lang code ' . $lang . 'is invalid');
        }
        // récuperation des données i18n
        if (isset($object['i18n'])) {
            $i18n = $object['i18n'];
            if ($i18n instanceof I18nModelInterface) {
                $i18n = [$i18n->getArrayCopy()];
            }
            try {
                // récuperationn de la donnée de langue
                $i18nData = static::getTranslateData($i18n, $lang, I18nLanguageCode::FR);
                // check du field demandé
                if ($field !== null && !isset($i18nData[$field])) {
                    trigger_error('field ' . $field . ' is not set in the i18n data');
                }
            } catch (\Exception $e) {
            }
        }
        // check du field demandé
        if ($field !== null && !isset($object[$field])) {
            trigger_error('field ' . $field . ' is not set in the object');
            return $i18nData[$field] ?? $default;
        }
        $field = $field ?: 'label';
        return $i18nData[$field] ?? $object[$field] ?? $default;
    }
}
