<?php

namespace Cms\Restful;

use League\Route\Http\Exception\BadRequestException;
use Move\Command\Command\CreateCommand;
use Move\Command\Command\DeleteCommand;
use Move\Command\Command\FetchAllCommand;
use Move\Command\Command\FetchCommand;
use Move\Command\Command\UpdateCommand;
use Move\Http\Controller\ResourceControllerInterface;
use Move\Http\RequestUtils;
use Move\Http\ResponseUtils;
use Move\Http\Strategy\ResponseFactory;
use POM\DomainObjectInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Zend\Diactoros\Response\EmptyResponse;

/**
 * Class AbstractResourceController
 * @package Cms\Restful
 */
abstract class AbstractResourceController implements ResourceControllerInterface
{

    /**
     * @param object $command
     * @return mixed
     */
    abstract protected function execCommand($command);

    /**
     * @param DomainObjectInterface $object
     * @param int|null $defaultTTL
     * @param int|null $archiveTTL
     * @return array
     */
    abstract protected function getObjectModifiedHeaders(
        DomainObjectInterface $object,
        $defaultTTL = null,
        $archiveTTL = null
    );

    /**
     * @param ServerRequestInterface $request
     * @param DomainObjectInterface $object
     * @return bool
     */
    abstract protected function isObjectNotModified(
        ServerRequestInterface $request,
        DomainObjectInterface $object
    );

    /**
     * @param DomainObjectInterface $object
     * @param int $code
     * @return ResponseFactory|ResponseInterface
     */
    abstract protected function makeObjectResponse(DomainObjectInterface $object, $code = 200);

    /**
     * @param array|\Iterator $collection
     * @param int $pageNum
     * @param int $byPage
     * @return ResponseFactory|ResponseInterface
     */
    abstract protected function makeCollectionResponse($collection, $pageNum = 0, $byPage = 20);


    /**
     * @param ServerRequestInterface $request
     * @return int
     * @throws BadRequestException
     */
    protected function getPageNum(ServerRequestInterface $request)
    {
        // récup des params
        $pageNum = RequestUtils::getQueryValue($request, 'page', 0);
        if (!is_numeric($pageNum) || $pageNum < -1) {
            throw new BadRequestException('page');
        }
        return $pageNum;
    }

    /**
     * @param ServerRequestInterface $request
     * @return array
     * @throws BadRequestException
     */
    protected function getSortCols(ServerRequestInterface $request)
    {
        $sortCols = RequestUtils::getQueryValue($request, 'sort', []);
        if (empty($sortCols)) {
            $sortCol = RequestUtils::getQueryValue($request, 'sort_col');
            $sortOrder = RequestUtils::getQueryValue($request, 'sort_order', 'ASC');
            if (!empty($sortCol)) {
                $sortCols = [$sortCol => $sortOrder];
            } else {
                $sortCols = [];
            }
        } elseif (!\is_array($sortCols)) {
            throw new BadRequestException('sort');
        }
        // check des valeurs de tri
        return array_filter($sortCols, function ($sortOrder) {
            return \in_array(strtoupper($sortOrder), ['DESC', 'ASC']);
        });
    }

    /**
     * @param ServerRequestInterface $request
     * @param string $sortCol
     * @return string|null ASC|DESC|null
     * @throws \League\Route\Http\Exception\BadRequestException
     */
    protected function getSortOrder(ServerRequestInterface $request, $sortCol)
    {
        $sortCols = $this->getSortCols($request);
        if (isset($sortCols[$sortCol])) {
            return strtoupper($sortCols[$sortCol]);
        }
        return null;
    }

    /**
     * @param ServerRequestInterface $request
     * @return array
     * @throws BadRequestException
     */
    protected function getFilters(ServerRequestInterface $request)
    {
        $filters = RequestUtils::getQueryValue($request, 'filter');
        if (empty($filters)) {
            $filterCol = RequestUtils::getQueryValue($request, 'filter_col');
            $filterValue = RequestUtils::getQueryValue($request, 'filter_value', true);
            if (!empty($filterCol)) {
                $filters = [$filterCol => $filterValue];
            } else {
                $filters = [];
            }
        } elseif (!\is_array($filters)) {
            throw new BadRequestException('filter');
        }
        return $filters;
    }

    /**
     * @param ServerRequestInterface $request
     * @param string $filterName
     * @return mixed|null
     * @throws \League\Route\Http\Exception\BadRequestException
     */
    protected function getFilterValue(ServerRequestInterface $request, $filterName)
    {
        $filters = $this->getFilters($request);
        if (isset($filters[$filterName])) {
            return $filters[$filterName];
        }
        // on cherche dans la querystring si non présent dans le filter array
        return RequestUtils::getQueryValue($request, $filterName);
    }

    /**
     * @param \Psr\Http\Message\ServerRequestInterface $request
     * @param mixed $result
     * @return \Psr\Http\Message\ResponseInterface|ResponseFactory|mixed
     * @throws \InvalidArgumentException
     */
    protected function applyNoCache(ServerRequestInterface $request, $result)
    {
        // check du param nocache
        $noCache = (bool)RequestUtils::getQueryValue($request, 'nocache', false);
        if ($noCache === true) {
            if ($result instanceof ResponseInterface) {
                // cleanup des headers
                $result = $result
                    ->withoutHeader('Cache-Control')
                    ->withoutHeader('Expires');
                // ajoute le nocache
                $result = ResponseUtils::withNoCache($result);
            } elseif ($result instanceof ResponseFactory) {
                $result->mergeHeaders(ResponseUtils::getNoCacheHeaders());
            }
        }

        return $result;
    }

    /**
     * @param ServerRequestInterface $request
     * @param \Psr\Http\Message\ResponseInterface $response
     * @param array $vars
     * @return mixed
     * @throws \InvalidArgumentException
     * @throws \League\Route\Http\Exception\BadRequestException
     */
    public function index(ServerRequestInterface $request, ResponseInterface $response, $vars = [])
    {
        // récup des params
        $pageNum = $this->getPageNum($request);
        $filters = $this->getFilters($request);
        $sortCols = $this->getSortCols($request);

        // traitement de l'iterateur
        $command = new FetchAllCommand($sortCols, $filters);
        $result = $this->execCommand($command);
        $result = $this->makeCollectionResponse($result, $pageNum);

        $result = $this->applyNoCache($request, $result);
        return $result;
    }

    /**
     * @param ServerRequestInterface $request
     * @param ResponseInterface $response
     * @param array $vars
     * @return ResponseFactory|EmptyResponse
     * @throws \InvalidArgumentException
     * @throws \UnexpectedValueException
     */
    public function show(ServerRequestInterface $request, ResponseInterface $response, $vars = [])
    {
        // exec command
        $command = new FetchCommand($vars['id']);
        $object = $this->execCommand($command);

        if (!$object) {
            $result = new EmptyResponse(
                204,
                ResponseUtils::getLastModifiedHeaders(time(), 300)
            );
        } elseif (!$object instanceof DomainObjectInterface) {
            throw new \UnexpectedValueException('object is not DomaineObjectInterface');
        } elseif ($this->isObjectNotModified($request, $object)) {
            $result = new EmptyResponse(
                304,
                $this->getObjectModifiedHeaders($object)
            );
        } else {
            $result = $this->makeObjectResponse($object);
        }

        $result = $this->applyNoCache($request, $result);
        return $result;
    }

    /**
     * @param ServerRequestInterface $request
     * @param \Psr\Http\Message\ResponseInterface $response
     * @param array $vars
     * @return mixed
     * @throws \UnexpectedValueException
     */
    public function create(ServerRequestInterface $request, ResponseInterface $response, $vars = [])
    {
        $dataset = $request->getParsedBody();

        // exec command
        $command = new CreateCommand($dataset ?: []);
        if (RequestUtils::getQueryValue($request, 'dryrun')) {
            $command = $command->withDryRun(true);
        }
        $object = $this->execCommand($command);
        if (!$object) {
            return new EmptyResponse();
        }
        if (!$object instanceof DomainObjectInterface) {
            throw new \UnexpectedValueException('object is not DomaineObjectInterface');
        }
        return $this->makeObjectResponse($object, 201);
    }

    /**
     * @param ServerRequestInterface $request
     * @param ResponseInterface $response
     * @param array $vars
     * @return mixed
     * @throws \UnexpectedValueException
     */
    public function update(ServerRequestInterface $request, ResponseInterface $response, $vars = [])
    {
        $dataset = $request->getParsedBody();

        // exec command
        $command = new UpdateCommand($vars['id'], $dataset ?: []);
        if (RequestUtils::getQueryValue($request, 'dryrun')) {
            $command = $command->withDryRun(true);
        }
        $object = $this->execCommand($command);
        if (!$object) {
            return new EmptyResponse();
        }
        if (!$object instanceof DomainObjectInterface) {
            throw new \UnexpectedValueException('object is not DomaineObjectInterface');
        }
        return $this->makeObjectResponse($object);
    }

    /**
     * @param ServerRequestInterface $request
     * @param ResponseInterface $response
     * @param array $vars
     * @return mixed
     * @throws \RuntimeException
     */
    public function delete(ServerRequestInterface $request, ResponseInterface $response, $vars = [])
    {
        // exec command
        $command = new DeleteCommand($vars['id']);
        if (RequestUtils::getQueryValue($request, 'dryrun')) {
            $command = $command->withDryRun(true);
        }
        $object = $this->execCommand($command);
        if (!$object) {
            throw new \RuntimeException('delete failed');
        }
        return $object;
    }
}
