<?php

namespace Move\Http\Strategy\Serializer;

use League\Plates\Engine;
use org\bovigo\vfs\vfsStream;
use PHPUnit\Framework\TestCase;
use Psr\Http\Message\ResponseInterface;

/**
 * Class PlatesAdapterSerializerTest
 * @package Move\Http\Strategy\Serializer
 */
class PlatesAdapterSerializerTest extends TestCase
{

    private $engine;
    private $minifier;
    private $folderName;
    private $errorDir;

    private $serializer;
    private $responseInterface;

    private $dataset            = ['key' => 'value'];
    private $correctDataset     = [
        'data' => ['key' => 'value'],
        'template' => 'test_tpl'
    ];

    private $templateContent    = '<tag><?php echo $key?></tag>';
    private $expectedTplResult  = '<tag>value</tag>';


    /**
     * @inheritdoc
     */
    public function setUp()
    {
        $this->setUpVirtualDirectory();

        $this->minifier = 'md5';
        $this->engine = new Engine($this->folderName);
    }

    protected function setUpVirtualDirectory()
    {
        $folder = 'tplfolder';
        vfsStream::setup($folder);
        vfsStream::create(
            [
                'error' => [
                    '50x.php' => '',
                ],
                'test_tpl.php' => $this->templateContent
            ]
        );

        $this->folderName = vfsStream::url($folder);
        $this->errorDir = vfsStream::url("$folder/error");
    }

    /**
     * prepare datas used for the testSerialize
     */
    protected function setupSerialize()
    {
        $this->responseInterface = $this->getMockBuilder(ResponseInterface::class)
            ->getMock();

        $this->serializer = new PlatesAdapterSerializer($this->engine);
    }

    /**
     * @covers PlatesAdapterSerializer::__construct
     */
    public function testConstruct()
    {
        $serializer = new PlatesAdapterSerializer($this->engine);
        $this->assertInstanceOf(SerializerInterface::class, $serializer);
        return $serializer;
    }

    /**
     * @covers PlatesAdapterSerializer::setErrorFolder
     * @depends testConstruct
     * @param PlatesAdapterSerializer $serializer
     */
    public function testSetErrorFolder(PlatesAdapterSerializer $serializer)
    {
        $res = $serializer->setErrorFolder($this->errorDir);
        $this->assertInstanceOf(PlatesAdapterSerializer::class, $res);
    }

    /**
     * @covers PlatesAdapterSerializer::setDefaultTemplateFolder
     * @depends testConstruct
     * @param PlatesAdapterSerializer $serializer
     */
    public function testSetDefaultTemplateFolder(PlatesAdapterSerializer $serializer)
    {
        $res = $serializer->setDefaultTemplateFolder(
            $this->folderName,
            $this->folderName
        );
        $this->assertInstanceOf(PlatesAdapterSerializer::class, $res);
    }

    /**
     * @covers PlatesAdapterSerializer::setHtmlMinifier
     * @depends testConstruct
     * @param PlatesAdapterSerializer $serializer
     */
    public function testSetHtmlMinifier(PlatesAdapterSerializer $serializer)
    {
        $minifier = function($str) {
            return preg_replace('#\s+<#','<',$str);
        };

        $res = $serializer->setHtmlMinifier($minifier);
        $this->assertInstanceOf(PlatesAdapterSerializer::class, $res);
    }

    /**
     * @covers PlatesAdapterSerializer::serialize
     * @depends testConstruct
     */
    public function testSerialize()
    {
        $this->setupSerialize();
        $serialized = $this->serializer->serialize(
            $this->correctDataset,
            $this->responseInterface
        );

        $this->assertEquals($this->expectedTplResult, $serialized);
    }

    /**
     * @covers PlatesAdapterSerializer::serialize
     * @depends testConstruct
     * @expectedException \UnexpectedValueException
     */
    public function testSerializeThrowsUnexpectedValueException()
    {
        $this->setupSerialize();

        $this->serializer->serialize(
            $this->dataset,
            $this->responseInterface
        );

    }

    /**
     * @covers PlatesAdapterSerializer::__invoke
     * @depends testSerialize
     */
    public function testInvoke()
    {
        $this->setupSerialize();

        $serialized = call_user_func(
            $this->serializer,
            $this->correctDataset,
            $this->responseInterface
        );

        $this->assertEquals($this->expectedTplResult, $serialized);
    }
}
