<?php


namespace Move\Image;

/**
 * Class Image
 * @package Move\Image
 */
class Image
{

    /** @var  int */
    protected $imageType;

    /** @var string */
    protected $mimeType;

    /** @var resource */
    protected $resource;

    /** @var bool */
    protected $animate;

    /** @var array */
    protected $imageinfos;

    /**
     * Image constructor.
     * @param string $raw
     * @throws \InvalidArgumentException
     */
    public function __construct(string $raw)
    {
        $this->resource = ImageUtils::createImage($raw);
        if (get_resource_type($this->resource) !== 'gd') {
            throw new \InvalidArgumentException('the resource must be a gd resource');
        }
        $this->animate = ImageUtils::isAnimated($raw);
        [, , $this->imageType] = getimagesizefromstring($raw, $this->imageinfos);
        $this->mimeType = image_type_to_mime_type($this->imageType);
    }

    /**
     * @return string
     */
    public function __toString()
    {
        return ImageUtils::imageToString($this->resource, $this->mimeType);
    }

    /**
     * @param int $quality
     * @return string
     */
    public function getJpegString(int $quality = null) : string
    {
        return ImageUtils::imageToString($this->resource, 'image/jpeg', $quality);
    }

    /**
     * @param int $quality
     * @return string
     */
    public function getPngString(int $quality = null) : string
    {
        return ImageUtils::imageToString($this->resource, 'image/png', $quality);
    }

    /**
     * @param int $pageWidth
     * @param int $pageHeight
     * @return Image
     * @throws \InvalidArgumentException
     * @throws \RuntimeException
     */
    public function scale(int $pageWidth, int $pageHeight = -1) : Image
    {
        $clone = clone $this;
        $clone->resource = ImageUtils::scaleImage($this->resource, $pageWidth, $pageHeight);
        $clone->animate = false; // pas d'animation si scale
        return $clone;
    }

    /**
     * @param array $size
     * @return Image
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     */
    public function scaleCut(array $size) : Image
    {
        $clone = clone $this;
        $clone->resource = ImageUtils::scaleImageCut($this->resource, $size);
        $clone->animate = false; // pas d'animation si scale
        return $clone;
    }

    /**
     * @param int $focusX
     * @param int $focusY
     * @param array $size
     * @return Image
     * @throws \RuntimeException
     * @throws \InvalidArgumentException
     */
    public function scaleWithFocus(int $focusX, int $focusY, array $size) : Image
    {
        $clone = clone $this;
        $clone->resource = ImageUtils::scaleImageWithFocus($this->resource, $focusX, $focusY, $size);
        $clone->animate = false; // pas d'animation si scale
        return $clone;
    }

    /**
     * @return string
     */
    public function getMimeType() : string
    {
        return $this->mimeType;
    }

    /**
     * @return int
     */
    public function getImageType() : int
    {
        return $this->imageType;
    }

    /**
     * @return bool
     */
    public function isAnimate() : bool
    {
        return $this->animate;
    }

    /**
     * @return \Move\Image\Iptc
     * @throws \InvalidArgumentException
     */
    public function getIptc() : Iptc
    {
        return Iptc::createFromInfo($this->imageinfos);
    }

    /**
     * @return resource
     */
    public function getResource() : resource
    {
        return $this->resource;
    }
}
