<?php


namespace Cms\Client;

/**
 * Class ClientIndexQueryBuilder
 * @package Cms\Client
 */
class ClientIndexQueryBuilder
{

    /** @var  array */
    private $sorts;

    /** @var int */
    private $pageNum;

    /** @var  array */
    private $filters;

    /** @var AbstractClient */
    private $client;

    /** @var array */
    private $onlyFilters;

    /** @var array */
    private $onlySortCols;

    /**
     * CategoryClientBuilder constructor.
     * @param AbstractClient $client
     * @param array $onlyFilters
     * @param array $onlySortCols
     */
    final public function __construct(AbstractClient $client, $onlyFilters = [], $onlySortCols = [])
    {
        $this->pageNum = 0;
        $this->filters = [];
        $this->sorts = [];
        $this->client = $client;
        // list d'exclusion
        $this->onlyFilters = $onlyFilters;
        $this->onlySortCols = $onlySortCols;
    }

    /**
     * @param int $pageNum
     * @return CmsApiIndexIterator
     */
    public function getIndex($pageNum)
    {
        $self = $this->setPageNum($pageNum);
        $results = $this->client->getIndexWithBuilder($self);
        return $results;
    }

    /**
     * @param string $sortCol
     * @param string $sortOrder
     * @return $this
     */
    public function setSortCol($sortCol, $sortOrder)
    {
        $this->sorts[$sortCol] = $sortOrder;
        return $this;
    }

    /**
     * @param string $sortCol
     * @return string ASC|DESC
     */
    public function getSortOrder($sortCol)
    {
        if (!empty($this->onlySortCols) && !\in_array($sortCol, $this->onlySortCols)) {
            throw new \InvalidArgumentException('sortCol ' . $sortCol . ' is invalid');
        }
        if (isset($this->sorts[$sortCol])) {
            return $this->sorts[$sortCol];
        }
        return null;
    }

    /**
     * @param string $filterName
     * @param mixed $filterValue
     * @return $this
     */
    public function setFilter($filterName, $filterValue)
    {
        if (!empty($this->onlyFilters) && !\in_array($filterName, $this->onlyFilters)) {
            throw new \InvalidArgumentException('filterName ' . $filterName . ' is invalid');
        }
        $this->filters[$filterName] = $filterValue;
        return $this;
    }

    /**
     * @param string $filterName
     * @return mixed|null
     */
    public function getFilter($filterName)
    {
        if (isset($this->filters[$filterName])) {
            return $this->filters[$filterName];
        }
        return null;
    }

    /**
     * @return int
     */
    public function getPageNum()
    {
        return $this->pageNum;
    }

    /**
     * @param int $pageNum
     * @return $this
     */
    public function setPageNum($pageNum)
    {
        $this->pageNum = $pageNum;
        return $this;
    }

    /**
     * @return array
     */
    public function getQueryParams()
    {
        return array_filter([
            'sort' => $this->sorts,
            'filter' => $this->filters,
            'page' => $this->pageNum,
        ]);
    }

    /**
     * @param array $sortCol
     * @return $this
     */
    public function addOnlySortCol($sortCol)
    {
        $this->onlySortCols[] = $sortCol;
        $this->onlySortCols = array_unique($this->onlySortCols);
        return $this;
    }

    /**
     * @param string $filterName
     * @return $this
     */
    public function addOnlyFilter($filterName)
    {
        $this->onlyFilters[] = $filterName;
        $this->onlyFilters = array_unique($this->onlyFilters);
        return $this;
    }
}
