<?php

namespace Move\Utils;

/**
 * Class TaggedItemCollection
 * @package Move\Utils
 */
class TaggedItemCollection implements \IteratorAggregate
{

    /**
     * @var array
     */
    private $banishTags;

    /**
     * @var array
     */
    private $items;

    /**
     * @var string
     */
    private $uniqueOn;


    /**
     * @param array $banishTags
     */
    public function __construct(array $banishTags = [])
    {
        $this->items = [];
        $this->banishTags = $banishTags;
    }


    /**
     * @param mixed $object
     * @param array $tags
     * @return \StdClass
     */
    final protected function getStdClassFromObject($object, array $tags)
    {
        $assetInternal = new \StdClass();
        $assetInternal->object = $object;
        $assetInternal->tags = $tags;
        return $assetInternal;
    }

    /**
     * @param \StdClass $class
     * @return array
     */
    final protected function getTagsFromStdClass($class)
    {
        if (\is_object($class)) {
            return $class->tags;
        }
        return [];
    }

    /**
     * @param \StdClass $class
     * @return mixed
     */
    final protected function getObjectFromStdClass($class)
    {
        if (\is_object($class)) {
            return $class->object;
        }
        return null;
    }

    /**
     * @return array
     */
    public function getBanishTags()
    {
        return $this->banishTags;
    }

    /**
     * @param TaggedItemCollection $itemCollection
     * @param bool $addBanish
     * @param bool $unique
     * @return $this
     */
    public function mergeWith(TaggedItemCollection $itemCollection, $addBanish = false, $unique = false)
    {
        // ajout des banish tags de l'autre collection
        $banishList = $itemCollection->getBanishTags();
        if ($addBanish === true && !empty($banishList)) {
            $this->banishByTags(
                array_unique(
                    array_merge($this->getBanishTags(), $banishList)
                )
            );
        }
        // ajoute les item de l'autre collection
        foreach ($itemCollection as $itemClass) {
            $tags = $this->getTagsFromStdClass($itemClass);
            $object = $this->getObjectFromStdClass($itemClass);
            $this->add($object, $tags, $unique);
        }
        return $this;
    }

    /**
     * Active le check de l'unicité sur la class
     */
    public function setUniqueOnClassName()
    {
        $this->uniqueOn = 'className';
    }

    /**
     * Active le check de l'unicité sur le premier tag
     */
    public function setUniqueOnFirstTag()
    {
        $this->uniqueOn = 'firstTag';
    }

    /**
     * Ajoute un nouvel objet à la liste
     * @param mixed $object
     * @param array $tags
     * @param bool $unique
     * @param bool $prepend
     * @return $this
     */
    public function add($object, array $tags = [], $unique = false, $prepend = false)
    {
        // si aucun tag passer on tag avec le nom de class
        $className = \get_class($object);
        $tags[] = $className;
        // dedup des tags
        $tags = array_unique($tags);
        // si unique on la supprime des tags class
        if ($unique === true) {
            $compareTo = $tags[0];
            if ($this->uniqueOn === 'className') {
                $compareTo = $className;
            }
            $this->removeInTags([$compareTo]);
        }
        // on check si la class est dans les banishs
        if (!array_intersect($tags, $this->banishTags)) {
            $assetInternal = $this->getStdClassFromObject($object, $tags);
            if ($prepend === true) {
                array_unshift($this->items, $assetInternal);
            } else {
                $this->items[] = $assetInternal;
            }
        }
        return $this;
    }

    /**
     * Supprime les assets dont le tag est dans $tags
     * @param array $tags
     * @return $this
     */
    public function removeInTags(array $tags)
    {
        $filtered = array_filter($this->items, function ($item) use ($tags) {
            $intersect = array_intersect($tags, $item->tags);
            return count($intersect) === 0;
        });
        $this->items = array_values($filtered);
        return $this;
    }

    /**
     * Renvoi la liste des object dont TOUS les tags sont $tags
     * @param array $tags
     * @return array
     */
    public function getByTags(array $tags)
    {
        $filtered = array_filter($this->items, function ($item) use ($tags) {
            return array_intersect($tags, $item->tags) == $tags;
        });
        return array_values(array_map([$this, 'getObjectFromStdClass'], $filtered));
    }

    /**
     * Renvoi la liste des objets dont UN des tag est $tag
     * @param string $tag
     * @return array
     */
    public function getByTag($tag)
    {
        $filtered = array_filter($this->items, function ($item) use ($tag) {
            return \in_array($tag, $item->tags);
        });
        return array_values(array_map([$this, 'getObjectFromStdClass'], $filtered));
    }

    /**
     * Remet à zéro la liste d'objets
     * @return $this
     */
    public function removeAll()
    {
        foreach ($this->items as &$asset) {
            $asset = null;
        }
        return $this;
    }

    /**
     * @return array
     */
    public function getAll()
    {
        return array_values(array_map([$this, 'getObjectFromStdClass'], $this->items));
    }

    /**
     * Liste de tags dont les objets ne seront pas ajoutable à la collection
     * @param array $tags
     * @return $this
     */
    public function banishByTags(array $tags)
    {
        $this->banishTags = $tags;
        // on épure au cas ou des element serai deja present
        foreach ($tags as $tag) {
            $this->removeInTags($tag);
        }
        return $this;
    }

    /**
     * Réinitialise le tableau des tags bannies
     * @return $this
     */
    public function resetBanishTags()
    {
        $this->banishTags = [];
        return $this;
    }

    /**
     * @inheritdoc
     */
    public function getIterator()
    {
        return new \ArrayIterator($this->items);
    }
}
