<?php

namespace Cms\Search;

use Move\Utils\Str;

/**
 * Trait ElasticQueryBuilderTrait
 * @package Cms\Search
 */
trait ElasticQueryBuilderTrait
{

    /** @var  array */
    protected $query_params = [
        'bool' => [
            'filter' => [],
        ],
    ];

    /** @var  array */
    protected $sourceFields;

    /** @var array */
    protected $aggs = [];

    /** @var array */
    protected $aggsFilter = [];

    /** @var int */
    protected $aggsSize = 15;

    /** @var array */
    protected $filters = [
        'searchText' => null,
    ];

    /** @var array */
    protected $functionScore = [];

    /**
     * @param array $fields
     * @return $this
     */
    public function setSourceFilter(array $fields)
    {
        $this->sourceFields = $fields;
        return $this;
    }

    /**
     * @return array|null
     */
    public function getSourceFilterAsArray()
    {
        if (!$this->sourceFields) {
            return null;
        }
        return $this->sourceFields;
    }

    /**
     * @param array $functions
     * @param array $opts
     * @return $this
     */
    public function setFunctionScore(array $functions, array $opts = [])
    {
        $this->functionScore = [$functions, $opts];
        return $this;
    }

    /**
     * @param array $function
     * @return $this
     */
    public function addFunctionScore(array $function)
    {
        if (!empty($this->functionScore)) {
            $this->functionScore[0][] = $function;
        }
        return $this;
    }

    /**
     * @param array $query
     * @return array
     */
    protected function applyFunctionScore(array $query) : array
    {
        if (!empty($this->functionScore)) {
            $functionScore = ElasticFunctionScoreFactory::createGlobalFunctionScore(
                $this->functionScore[0],
                $query,
                $this->functionScore[1]
            );
            return $functionScore;
        }
        return $query;
    }

    /**
     * @param string $aggtype
     * @param mixed $searchTerm
     * @param bool $add
     * @return $this
     */
    public function hasAggregate(string $aggtype, $searchTerm = true, bool $add = true)
    {
        if ($add === true) {
            $this->aggs[$aggtype] = $searchTerm;
        } else {
            unset($this->aggs[$aggtype]);
        }
        return $this;
    }

    /**
     * @param string $aggtype
     * @param array $filterQuery
     * @param bool $add
     * @return $this
     */
    public function hasAggregateFilters(string $aggtype, array $filterQuery = [], bool $add = true)
    {
        if ($add === true) {
            $this->aggsFilter[$aggtype] = $filterQuery;
        } else {
            unset($this->aggsFilter[$aggtype]);
        }
        return $this;
    }

    /**
     * @param mixed $facetName
     * @param array $facetOpts
     * @param array $facetFitlers
     * @return $this
     */
    public function addFacet(
        $facetName,
        array $facetOpts,
        array $facetFitlers = []
    ) {
        $this->hasAggregate($facetName, ...$facetOpts);
        $aggFilter['bool'] = [
            'must' => array_values(array_filter($facetFitlers)),
        ];
        if (!empty($aggFilter['bool']['must'])) {
            $this->hasAggregateFilters($facetName, $aggFilter);
        }
        return $this;
    }

    /**
     * @param int $aggsSize
     * @return $this
     */
    public function setAggsSize(int $aggsSize)
    {
        $this->aggsSize = $aggsSize;
        return $this;
    }

    /**
     * @param string $textSearch
     * @return $this
     */
    public function setSearchTerm(string $textSearch)
    {
        $this->filters['searchText'] = $textSearch;
        return $this;
    }


    /**
     * @param string $path
     * @param array $subQuery
     * @param bool|string|null $innerHits
     * @param float|null $boost
     * @return array
     */
    public function getNestedQuery(string $path, array $subQuery, $innerHits = null, $boost = null) : array
    {
        $nestedQuery = [
            'nested' => [
                'path' => $path,
                'query' => $subQuery,
            ],
        ];
        if ($innerHits === true) {
            $nestedQuery['nested']['inner_hits'] = new \stdClass();
        }
        if (\is_array($innerHits)) {
            $nestedQuery['nested']['inner_hits'] = $innerHits;
        }
        if ($boost !== null) {
            $nestedQuery['nested']['boost'] = $boost;
        }
        return $nestedQuery;
    }

    /**
     * @param array $query
     * @param string $operator
     * @param int|null $minimumMatch
     * @return array
     */
    public function getBoolQuery(
        array $query,
        string $operator = 'must',
        int $minimumMatch = null
    ) : array {
        $boolQuery = [
            'bool' => [
                $operator => $query,
            ],
        ];
        if ($minimumMatch && $operator === 'should') {
            $boolQuery['bool']['minimum_should_match'] = $minimumMatch;
        }
        return $boolQuery;
    }

    /**
     * @param string $colName
     * @param string $string
     * @return array
     */
    public function getMatchQuery(string $colName, string $string) : array
    {
        return [
            'match' => [
                $colName => $string,
            ],
        ];
    }

    /**
     * @param string $colName
     * @param array $terms
     * @param float|null $boost
     * @return array
     */
    public function getTermsQuery(string $colName, array $terms, $boost = null) : array
    {
        $boolQuery = [];
        if (count($terms) > 1024) {
            foreach (array_chunk($terms, 1024) as $chunk) {
                $boolQuery[] = $this->getTermsQuery($colName, $chunk, $boost);
            }
            return $this->getBoolQuery($boolQuery, 'should', !$boost ? 1 : null);
        }
        $terms = array_values($terms);
        if ($colName === '_id') {
            $query = [
                'ids' => [
                    'values' => $terms,
                ],
            ];
        } elseif (count($terms) === 1) {
            $query = $this->getTermQuery($colName, current($terms), $boost);
        } else {
            $query = [
                'terms' => [
                    $colName => $terms,
                ],
            ];
            if ($boost) {
                $query['terms']['boost'] = $boost;
            }
        }
        return $query;
    }

    /**
     * @param array $colNames
     * @param array $terms
     * @param float|null $boost
     * @return array|null
     */
    public function getMultiTermsQuery(array $colNames, array $terms, $boost = null)
    {
        $boolQuery = [];
        if (count($terms) > 1024) {
            foreach (array_chunk($terms, 1024) as $chunk) {
                $boolQuery[] = $this->getMultiTermsQuery($colNames, $chunk, $boost);
            }
            return $this->getBoolQuery($boolQuery, 'should', !$boost ? 1 : null);
        }
        $terms = array_values($terms);

        foreach ($colNames as $colName) {
            $termsQuery = [$colName => $terms];
            if ($boost) {
                $termsQuery['boost'] = $boost;
            }
            $query[]['terms'] = $termsQuery;
        }
        if (!empty($query)) {
            return $this->getBoolQuery($query, 'should');
        }
        return null;
    }

    /**
     * @param string $colName
     * @param string $term
     * @param float|null $boost
     * @return array
     */
    public function getTermQuery(string $colName, string $term, $boost = null) : array
    {
        $query = [
            'term' => [
                $colName => [
                    'value' => $term,
                ],
            ],
        ];
        if ($boost) {
            $query['term'][$colName]['boost'] = $boost;
        }
        return $query;
    }

    /**
     * @param array $aggsQuery
     * @param string $aggName
     * @param array $aggFragment
     * @param string|null $nestedPath
     */
    protected function appendAggFragment(
        array &$aggsQuery,
        string $aggName,
        array $aggFragment,
        string $nestedPath = null
    ) {
        if ($nestedPath) {
            $aggsQuery[$aggName . '_nested'] = [
                'nested' => [
                    'path' => $nestedPath,
                ],
                'aggs' => [],
            ];
            $this->appendAggFragment(
                $aggsQuery[$aggName . '_nested']['aggs'],
                $aggName,
                $aggFragment
            );
        } elseif (!empty($this->aggsFilter[$aggName])) {
            $aggsQuery[$aggName . '_filter'] = [
                'filter' => $this->aggsFilter[$aggName],
                'aggs' => [],
            ];
            // construction d'un nouveau builder
            $builder = clone $this;
            $builder->aggsFilter[$aggName] = null;
            $builder->appendAggFragment(
                $aggsQuery[$aggName . '_filter']['aggs'],
                $aggName,
                $aggFragment
            );
        } else {
            $aggsQuery[$aggName] = $aggFragment;
        }
    }

    /**
     * @param array $aggsQuery
     * @param string $aggName
     * @param string $fieldName
     * @param int $size
     * @param array $order
     * @param string|null $nestedPath
     */
    protected function appendSimpleTermAgg(
        array &$aggsQuery,
        string $aggName,
        string $fieldName,
        int $size,
        array $order = [],
        string $nestedPath = null
    ) {
        if (isset($this->aggs[$aggName])) {
            $aggFragment = [
                'terms' => [
                    'field' => $fieldName,
                    'size' => $size,
                    'order' => $order ?: ['_count' => 'desc'],
                ],
            ];
            $this->appendAggFragment($aggsQuery, $aggName, $aggFragment, $nestedPath);
        }
    }

    /**
     * @param string $nestedPath
     * @param array $filter
     * @param array $aggData
     * @param string|null $aggName
     * @return array
     */
    protected function getAggsNestedQuery(
        string $nestedPath,
        array $filter = [],
        array $aggData = [],
        string $aggName = null
    ) : array {
        $aggName = $aggName ?: 'division';
        $agg = [
            'nested' => [
                'path' => $nestedPath,
            ],
            'aggs' => [
                $aggName => [],
            ],
        ];
        if (!empty($filter)) {
            $agg['aggs'][$aggName]['filter'] = $filter;
        }
        if (!empty($aggData)) {
            $agg['aggs'][$aggName]['aggs'] = $aggData;
        }
        return $agg;
    }

    /**
     * @param string $fieldName
     * @param string|null $lt
     * @param string|null $gt
     * @param bool $ltequal
     * @param bool $gtequal
     * @param null|string $boost
     * @return array|null
     */
    public function getRangeQuery(
        string $fieldName,
        $lt = null,
        $gt = null,
        bool $ltequal = true,
        bool $gtequal = true,
        $boost = null
    ) {
        if ($lt) {
            $ranges[$ltequal ? 'lte' : 'lt'] = $lt;
        }
        if ($gt) {
            $ranges[$gtequal ? 'gte' : 'gt'] = $gt;
        }
        if (($gt || $lt) && $boost) {
            $ranges['boost'] = $boost;
        }
        return !empty($ranges) ? [
            'range' => [
                $fieldName => $ranges,
            ],
        ] : null;
    }

    /**
     * @return string
     */
    public function __toString()
    {
        try {
            $query = $this->getQueryAsArray();
            $aggs = $this->getAggsAsArray();
            if ($aggs) {
                $query['aggs'] = $aggs;
            }
            return Str::toJson($query, JSON_PRETTY_PRINT);
        } catch (\Exception $e) {
            return '';
        }
    }

    /**
     * @return array
     */
    abstract public function getAggsAsArray() : array;

    /**
     * @return array
     */
    abstract public function getQueryAsArray() : array;
}
