<?php

namespace Cms\Serializer;

use function Cms\DashParser\create_document;
use Cms\DashParser\Entity\EntityDocument;
use Cms\DashParser\Entity\EntityInterface;
use function Cms\DashParser\load_html;

/**
 * Class HtmlEntityDocumentConverter
 * @package Cms\Serializer
 */
class HtmlEntityDocumentConverter extends \Cms\DashParser\Serializer\HtmlEntityDocumentConverter
{

    /** @var null|string */
    private $fallbackTxt;

    /**
     * HtmlEntityDocumentConverter constructor.
     * @param string|null $fallbackTxt
     */
    public function __construct(string $fallbackTxt = null)
    {
        $this->fallbackTxt = $fallbackTxt ?: 'Ce navigateur ne supporte pas les fichiers PDF. '
        . '<a href="%s">Veuillez télécharger le PDF pour le voir</a>';
    }

    /**
     * @param \DOMDocument $doc
     * @param \Cms\DashParser\Entity\EntityInterface $entity
     * @param null $nodeText
     * @return \DOMElement|null
     */
    public function getEntityNode(\DOMDocument $doc, EntityInterface $entity, $nodeText = null)
    {
        // check type de node
        if (!$entity instanceof EntityDocument) {
            return null;
        }

        // object (HTML5)
        $ObjectNode = parent::getEntityNode($doc, $entity, $nodeText);
        if ($ObjectNode === null) {
            return $ObjectNode;
        }

        // creation de l'iframe de fallback
        $fallbackDoc = create_document();
        $fallbackTxt = sprintf($this->fallbackTxt, $entity->getUrl());
        load_html($fallbackDoc, $fallbackTxt);
        $fallbackNode = $doc->importNode($fallbackDoc->firstChild, true);
        $ObjectNode->appendChild($fallbackNode);

        // container for responsive
        $divNode = $doc->createElement('div');
        $divNode->setAttribute('class', 'embed-container');
        $divNode->appendChild($ObjectNode);


        $linkNode = $doc->createElement('a', 'Afficher en plein écran');
        $linkNode->setAttribute('href', $entity->getUrl());
        $linkNode->setAttribute('target', '_blank');

        $wrapperNode = $doc->createElement('div');
        $wrapperNode->setAttribute('style', 'text-align: right');
        $wrapperNode->appendChild($divNode);
        $wrapperNode->appendChild($linkNode);

        return $wrapperNode;
    }
}
