<?php

namespace Move\Embed\Adapter;

use Move\Embed\EmbedFactory;
use Move\Embed\EmbedModel;
use Move\Embed\Normalizer\NormalizerInterface;
use Move\Http\Client\HttpRequestOptions;
use Move\Http\Middleware\BodyParser;

/**
 * Class Oembed
 * @package Embed\Adapter
 */
class Oembed implements AdapterInterface
{

    /**
     * @var string
     */
    private $endpoint;

    /**
     * @var array
     */
    private $defaultParams;

    /**
     * @var NormalizerInterface
     */
    private $normalizer;


    /**
     * @param string $endpoint (http://flickr.com/services/oembed)
     * @param array $defaultParams (['format'=>'json'])
     */
    public function __construct($endpoint, array $defaultParams = [])
    {
        $this->endpoint = $endpoint;
        $this->defaultParams = array_merge([
            'format' => 'json',
        ], $defaultParams);
    }

    /**
     * Un provider peut être valide pour plusieurs URI (ex : youtube.com, youtu.be)
     * mais l'oembed fonctionne parfois qu'avec un seul format d'URL (ex : youtube.com).
     * Cette fonction remédie à ce problème en normalisant les URI à un format standard
     * pour le provider.
     * @param string $sourceUri
     * @return string
     */
    public function normalizeUri($sourceUri) : string
    {
        if (!$this->normalizer) {
            return $sourceUri;
        }
        return $this->normalizer->normalizeStr($sourceUri);
    }

    /**
     * Renvoi les données de la source sous forme de model
     * @param string $sourceUri
     * @param array $params
     * @return EmbedModel|null
     */
    public function getEmbedModel($sourceUri, array $params = [])
    {
        $sourceUri = $this->normalizeUri($sourceUri);
        $content = $this->call($sourceUri, $params);
        if (\is_array($content)) {
            $embedModel = new EmbedModel($content);
            $embedModel->source_url = $sourceUri;
            // cas des format photo
            if (isset($content['type']) && $content['type'] === 'photo' && !$embedModel->html) {
                if (empty($content['url'])) {
                    return null;
                }
                $embedModel->html = '<img src="' . $content['url'] . '" width="' . $content['width'] . '" height="' . $content['height'] . '" />';
            }
            if (!$embedModel->html) {
                return null;
            }
            return $embedModel;
        }
        return null;
    }

    /**
     * @param $sourceUri
     * @param array $params
     * @return array|null
     */
    protected function call($sourceUri, array $params = [])
    {
        try {
            $client = EmbedFactory::createClient(parse_url($this->endpoint, PHP_URL_HOST));
            $results = $client->request('GET', [parse_url($this->endpoint, PHP_URL_PATH)], [
                HttpRequestOptions::QUERY_PARAMS => array_merge($this->defaultParams, $params, [
                    'url' => $sourceUri,
                ]),
            ]);
            $resultParsed = (new BodyParser())->parseMessage($results);
            return $resultParsed;
        } catch (\Exception $e) {
            trigger_error((string)$e, E_USER_WARNING);
        }
        return null;
    }

    /**
     * @param NormalizerInterface $normalizer
     */
    public function setNormalizer(NormalizerInterface $normalizer)
    {
        $this->normalizer = $normalizer;
    }
}
