<?php

namespace Move\Filter;

use Move\Specification\EmptySpecification;
use Move\Specification\NotSpecification;
use Move\Specification\PhpFilterVarSpecification;
use Move\Specification\SpecificationInterface;

/**
 * Class FilterAbsract
 * @package Move\Filter
 */
class Filter
{

    /**
     * @var array
     */
    private $sanitizers = [];

    /**
     * @var SpecificationInterface[]
     */
    private $specs = [];

    /**
     * @var array
     */
    private $modifiers = [];

    /**
     * @var SpecificationInterface[]
     */
    private $failed_specs = [];

    /**
     * @var Filter
     */
    private $pipe_filter;

    /**
     * @var array
     */
    private $failed_values;


    /**
     * Filter constructor.
     * @param array $filters
     * @param array $modifiers
     * @param array $sanitizers
     * @param Filter|null $pipe
     */
    public function __construct(array $filters = [], array $modifiers = [], array $sanitizers = [], $pipe = null)
    {
        // ajout des filtre
        foreach ($filters as $fieldName => $filter) {
            $this->filter($fieldName, $filter);
        }

        // ajout des modifiers
        foreach ($modifiers as $fieldName => $modifier) {
            $this->modify($fieldName, $modifier);
        }

        // ajout des sanitizers
        foreach ($sanitizers as $fieldName => $sanitizer) {
            $this->sanitize($fieldName, $sanitizer);
        }

        // ajout du pipe
        if (!empty($pipe)) {
            $this->pipe($pipe);
        }
    }

    /**
     * @param string $fieldName
     * @param mixed $filter
     * @param array|string $opts
     * @return $this
     */
    public function filter($fieldName, $filter, $opts = null)
    {
        // creation de la spec si non passé
        if (\is_bool($filter)) {
            $empty = new EmptySpecification();
            if ($filter === true) {
                $filter = new NotSpecification($empty);
            } else {
                $filter = $empty;
            }
        } elseif (!$filter instanceof SpecificationInterface) {
            if (\is_callable($filter)) {
                $opts = $filter;
                $filter = FILTER_CALLBACK;
            }
            $filter = new PhpFilterVarSpecification($filter, [
                'flags' => !\is_array($opts) && $filter !== FILTER_CALLBACK ? $opts : null,
                'options' => $filter === FILTER_CALLBACK ? $opts : (\is_array($opts) ? $opts : []),
            ]);
        }

        $this->spec($fieldName, $filter);

        return $this;
    }

    /**
     * Ajoute une spec sur un champs, si une spec existe deja ou l'ajoute comme 'andSpec'
     * @param string $fieldName
     * @param SpecificationInterface $spec
     * @return $this
     */
    final public function spec($fieldName, SpecificationInterface $spec)
    {
        if (!empty($this->specs[$fieldName])) {
            $this->specs[$fieldName] = $this->specs[$fieldName]->andSpec($spec);
        } else {
            $this->specs[$fieldName] = $spec;
        }
        return $this;
    }

    /**
     * @param string $fieldName
     * @param callable $modifier
     * @param array $args
     * @return $this
     */
    public function modify($fieldName, callable $modifier, array $args = [])
    {
        $this->modifiers[$fieldName] = [
            'callback' => $modifier,
            'args' => $args,
        ];
        return $this;
    }

    /**
     * @param string $fieldName
     * @param mixed $sanitizer
     * @param array $opts
     * @return $this
     */
    public function sanitize($fieldName, $sanitizer, array $opts = [])
    {
        $this->sanitizers[$fieldName] = [
            'filter' => $sanitizer,
            'flags' => !\is_array($opts) ? $opts : null,
            'options' => \is_array($opts) ? $opts : [],
        ];
        return $this;
    }

    /**
     * Permet de piper un autre filtre sur un champs
     * @param Filter $filter
     * @return $this
     */
    public function pipe(Filter $filter)
    {
        $this->pipe_filter = $filter;

        return $this;
    }

    /**
     * @param mixed $postData
     * @param string|null $fieldName
     * @return mixed
     */
    public function __invoke($postData, $fieldName = null)
    {
        return $this->process($postData, $fieldName);
    }

    /**
     * @param array $postData
     * @param string|null $fieldName
     * @return mixed
     */
    public function process($postData, $fieldName = null)
    {
        if (!\is_array($postData)) {
            throw new \InvalidArgumentException('input data must be array');
        }
        // restart
        $this->failed_specs = [];
        $this->failed_values = [];

        if (!empty($fieldName)) {
            if (isset($postData[$fieldName])) {
                $postData = array_intersect_key($postData, array_flip([$fieldName]));
            } else {
                $postData = [$fieldName => null];
            }
        }

        // application des cleaner
        $postDataToSan = array_intersect_key($postData, $this->sanitizers ?: []);
        $postDataSanitize = array_merge($postData, filter_var_array($postDataToSan, $this->sanitizers) ?: []);

        // application des specs
        $postDataFiltered = [];
        foreach ($this->specs as $field => $spec) {
            // si le filtre est applicable
            if (isset($postDataSanitize[$field])) {
                $postDataFiltered[$field] = false;
                if ($spec->isSatisfiedBy($postDataSanitize[$field])) {
                    $postDataFiltered[$field] = $postDataSanitize[$field];
                } else {
                    $this->failed_values[$field] = $postDataSanitize[$field];
                    $this->failed_specs[$field] = $spec;
                }
            } else {
                if (!$spec->isSatisfiedBy(null)) {
                    $this->failed_values[$field] = null;
                    $this->failed_specs[$field] = $spec;
                }
            }
        }
        $postDataFiltered = array_merge($postDataSanitize, $postDataFiltered);

        // application des modifiers
        foreach ($postDataFiltered as $key => &$dataValue) {
            if (!empty($this->modifiers[$key])) {
                $args = array_merge([$dataValue], $this->modifiers[$key]['args'], [$postDataFiltered]);
                $dataValue = \call_user_func_array($this->modifiers[$key]['callback'], $args);
            }
        }

        // application du pipe
        if (!empty($this->pipe_filter)) {
            $postDataPiped = $this->pipe_filter->process($postDataFiltered, $fieldName);
            if (!empty($fieldName)) {
                $postDataFiltered = array_merge($postDataFiltered, [$fieldName => $postDataPiped]);
            } else {
                $postDataFiltered = $postDataPiped;
            }
            $this->failed_values = array_merge($this->failed_values, $this->pipe_filter->getFailedValues());
            $this->failed_specs = array_merge($this->failed_specs, $this->pipe_filter->getFailedSpecs());
        }

        // renvoi des data
        if ($fieldName !== null) {
            return $postDataFiltered[$fieldName];
        }
        return $postDataFiltered;
    }

    /**
     * @return SpecificationInterface[]
     */
    public function getFailedSpecs()
    {
        return $this->failed_specs ?: [];
    }

    /**
     * @return array
     */
    public function getFailedValues()
    {
        return $this->failed_values ?: [];
    }
}
