<?php


namespace Move\Command;

use Interop\Container\ContainerInterface;
use League\Tactician\CommandBus;
use League\Tactician\Container\ContainerLocator;
use League\Tactician\Handler\CommandHandlerMiddleware;
use League\Tactician\Handler\CommandNameExtractor\ClassNameExtractor;
use League\Tactician\Handler\CommandNameExtractor\CommandNameExtractor;
use League\Tactician\Handler\MethodNameInflector\HandleClassNameWithoutSuffixInflector;
use League\Tactician\Handler\MethodNameInflector\HandleInflector;
use League\Tactician\Handler\MethodNameInflector\MethodNameInflector;

/**
 * Class TacticianBusFactory
 * @package Move\Command
 */
class TacticianBusFactory
{

    /**
     * @param ContainerInterface $container
     * @param array $commandMap
     * @param CommandNameExtractor|null $extractor
     * @param MethodNameInflector|null $inflector
     * @return CommandHandlerMiddleware
     */
    public static function createContainerHandler(
        ContainerInterface $container,
        array $commandMap,
        CommandNameExtractor $extractor = null,
        MethodNameInflector $inflector = null
    ) : CommandHandlerMiddleware {

        // generation du locator (recherche des class)
        $locator = new ContainerLocator(
            $container,
            $commandMap
        );
        // creation de middleware
        $commandHandlerMiddleware = new CommandHandlerMiddleware(
            $extractor ?: new ClassNameExtractor(),
            $locator,
            $inflector ?: new HandleInflector()
        );

        return $commandHandlerMiddleware;
    }

    /**
     * @param ContainerInterface $container
     * @param array $commandToHandlerMap
     * @param MethodNameInflector $inflector
     * @param array $middlewareList
     * @return CommandBusInterface
     */
    public static function containerMap(
        ContainerInterface $container,
        array $commandToHandlerMap,
        MethodNameInflector $inflector = null,
        array $middlewareList = []
    ) : CommandBusInterface {
        // creation de middleware
        $commandHandlerMiddleware = self::createContainerHandler(
            $container,
            $commandToHandlerMap,
            null,
            $inflector
        );
        // generation du commandBus
        return new TactitianBusAdapter(
            new CommandBus(array_merge($middlewareList, [$commandHandlerMiddleware]))
        );
    }

    /**
     * @param ContainerInterface $container
     * @param array $commandList
     * @param string $handlerClass
     * @param array $middlewareList
     * @return CommandBusInterface
     */
    public static function mapToClassHandler(
        ContainerInterface $container,
        array $commandList,
        string $handlerClass,
        array $middlewareList = []
    ) : CommandBusInterface {
        $classMap = array_combine(
            $commandList,
            array_fill(0, count($commandList), $handlerClass)
        );

        // creation de middleware
        $commandHandlerMiddleware = self::createContainerHandler(
            $container,
            $classMap,
            null,
            new HandleClassNameWithoutSuffixInflector()
        );

        // generation du commandBus
        return new TactitianBusAdapter(
            new CommandBus(array_merge($middlewareList, [$commandHandlerMiddleware]))
        );
    }
}
