<?php


namespace Cms\Extension;

use Cms\Client\Article\Model\CategoryRestfulModel;
use Cms\Client\Command\Article\Command\FetchArticleCategoryCommand;
use Cms\Client\Service\ArticleService;
use Cms\LinkResource\LinkResourceModel;
use Cms\LinkResource\LinkResourceType;
use Cms\LinkResource\Provider\CMS\CmsProvider;
use Cms\Model\Article\Category\CategoryPluginType;
use Cms\Plugin\Article\Category\PluginColorConfig;
use League\Plates\Engine;
use League\Plates\Extension\ExtensionInterface;

/**
 * Class CategoryPluginColorExtension
 * @package Cms\Extension
 */
class CategoryPluginColorExtension implements ExtensionInterface
{

    /** @var ArticleService */
    private $articleService;

    /** @var null */
    private $scopeId;

    /**
     * CategoryPluginColorExtension constructor.
     * @param ArticleService $articleService
     * @param null $scopeId
     */
    public function __construct(ArticleService $articleService, $scopeId = null)
    {
        $this->articleService = $articleService;
        $this->scopeId = $scopeId;
    }

    /**
     * @param Engine $engine
     */
    public function register(Engine $engine)
    {
        $engine->registerFunction('color_name', [$this, 'getColorName']);
        $engine->registerFunction('color_hex', [$this, 'getColorHex']);
    }

    /**
     * @param mixed $rubrique
     * @return CategoryRestfulModel|null
     */
    protected function getRubriqueModel($rubrique)
    {
        // si resource link
        if ($rubrique instanceof LinkResourceModel
            && (string)$rubrique->link_type === LinkResourceType::ARTICLE_CATEGORY
            && $rubrique->link_provider === CmsProvider::IDENTIFIER
        ) {
            $rubrique = $rubrique->link_ref;
        }

        // recherche par id
        if (is_numeric($rubrique)) {
            $command = new FetchArticleCategoryCommand(null, $rubrique);
            $rubrique = $this->articleService->getCategoryCommandBus()->handle($command);
        } // recherche par slug
        elseif (\is_string($rubrique) && !empty($this->scopeId)) {
            $command = new FetchArticleCategoryCommand($rubrique);
            $command->setScopeId($this->scopeId);
            $rubrique = $this->articleService->getCategoryCommandBus()->handle($command);
        } elseif (\is_array($rubrique)) {
            $rubrique = (new CategoryRestfulModel())->populate($rubrique, true);
        }

        if ($rubrique instanceof CategoryRestfulModel) {
            return $rubrique;
        }
        return null;
    }

    /**
     * @param CategoryRestfulModel $rubrique
     * @return \Cms\Model\AbstractConfigModel|null
     */
    protected function getColorPluginConfig(CategoryRestfulModel $rubrique)
    {
        $plugins = $rubrique->plugins;
        foreach ($plugins as $plugin) {
            if ((string)$plugin->type === CategoryPluginType::COLOR
                && $plugin->config instanceof PluginColorConfig
            ) {
                return $plugin->config;
            }
        }
        return null;
    }

    /**
     * @param mixed $rubrique
     * @return string|null
     */
    public function getColorName($rubrique)
    {
        $rubrique = $this->getRubriqueModel($rubrique);

        // récuperation des plugins
        if ($rubrique !== null) {
            $plugin = $this->getColorPluginConfig($rubrique);
            if ($plugin instanceof PluginColorConfig) {
                return $plugin->colorCodeName;
            }
        }
        return null;
    }

    /**
     * @param mixed $rubrique
     * @return string|null
     */
    public function getColorHex($rubrique)
    {
        $rubrique = $this->getRubriqueModel($rubrique);

        // récuperation des plugins
        if ($rubrique !== null) {
            $plugin = $this->getColorPluginConfig($rubrique);
            if ($plugin instanceof PluginColorConfig) {
                return $plugin->colorCodeHex;
            }
        }
        return null;
    }
}
