<?php


namespace Move\Http\Client;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Exception\ServerException;
use GuzzleHttp\Promise\RejectedPromise;
use GuzzleHttp\RequestOptions;
use Move\Http\Client\Exception\HttpClientException;
use Move\Http\Client\Exception\HttpRequestException;
use Move\Http\Client\Exception\HttpServerException;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\NullLogger;

/**
 * Class GuzzleClientAdapter
 * @package Move\Http\Client
 */
class GuzzleClientAdapter extends HttpClientAbstract implements LoggerAwareInterface
{
    use LoggerAwareTrait;

    /** Guzzle option key */
    public const GUZZLE_OPT_KEY = 'guzzle';

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @var array
     */
    private $defaultHeaders;

    /**
     * @var
     */
    private $requestHost;

    /**
     * @var array
     */
    private $defaultOptions;

    /**
     * GuzzleClientAdapter constructor.
     * @param ClientInterface $client
     * @param $requestHost
     * @param array $defaultOptions
     * @throws \InvalidArgumentException
     */
    public function __construct(
        ClientInterface $client,
        string $requestHost,
        array $defaultOptions = []
    ) {
        $this->client = $client;
        $this->requestHost = $requestHost;
        $this->defaultOptions = $defaultOptions;

        // check opts
        $this->validateOptions($defaultOptions);

        // assign des headers par default
        $this->defaultHeaders = [];
        if (isset($defaultOptions[HttpRequestOptions::HEADERS])) {
            $this->defaultHeaders = $defaultOptions[HttpRequestOptions::HEADERS];
        }

        // ajout du logger
        $this->setLogger(new NullLogger());
    }

    /**
     * @param array $options
     * @return $this
     * @throws \InvalidArgumentException
     */
    protected function validateOptions($options)
    {
        foreach ($options as $key => $value) {
            if ($key !== self::GUZZLE_OPT_KEY && !HttpRequestOptions::isValid($key)) {
                throw new \InvalidArgumentException("option key is invalid : $key");
            }
        }
        return $this;
    }

    /**
     * @param array $options
     * @param string $optionName
     * @param null|mixed $defaultValue
     * @return mixed|null
     * @throws \InvalidArgumentException
     * @throws \UnexpectedValueException
     */
    protected function readOption($options, $optionName, $defaultValue = null)
    {
        // check opts
        $this->validateOptions($options);

        // récup
        $value = $defaultValue;
        if (!HttpRequestOptions::isValid($optionName)) {
            throw new \UnexpectedValueException('optionName is an invalid option');
        }
        if (isset($options[$optionName])) {
            $value = $options[$optionName];
        }
        if (isset($this->defaultOptions[$optionName])) {
            $value = $this->defaultOptions[$optionName];
        }
        return $value;
    }

    /**
     * @param array $options
     * @return array
     * @throws \InvalidArgumentException
     * @throws \UnexpectedValueException
     */
    protected function convertOptionsForGuzzle(array $options) : array
    {
        // init tableau options pour guzzle
        $guzzleOptions = [];

        // ajout de la queryString
        if (($queryParams = $this->readOption($options, HttpRequestOptions::QUERY_PARAMS, []))
            && \is_array($queryParams)
        ) {
            // trie avant envoi
            ksort($queryParams);
            $guzzleOptions = [
                RequestOptions::QUERY => $queryParams,
            ];
        }

        // ajout des headers
        $headers = $this->defaultHeaders;
        if (isset($options[HttpRequestOptions::HEADERS])) {
            $headers = array_merge($headers, $options[HttpRequestOptions::HEADERS]);
        }
        $guzzleOptions[RequestOptions::HEADERS] = $headers;

        // ajout le body content
        if (isset($options[HttpRequestOptions::BODY_CONTENT])) {
            $bodyContentType = RequestOptions::JSON;
            if (array_key_exists('content-type', array_change_key_case($headers))) {
                $bodyContentType = RequestOptions::BODY;
            }
            if (isset($options[HttpRequestOptions::MULTIPART])) {
                $bodyContentType = RequestOptions::MULTIPART;
            } elseif (isset($options[HttpRequestOptions::FORM_PARAMS])) {
                $bodyContentType = RequestOptions::FORM_PARAMS;
            }
            $guzzleOptions[$bodyContentType] = $options[HttpRequestOptions::BODY_CONTENT];
        }

        // merge avec les autre options
        if (isset($options[self::GUZZLE_OPT_KEY]) && \is_array($options[self::GUZZLE_OPT_KEY])) {
            $guzzleOptions = array_merge($guzzleOptions, $options[self::GUZZLE_OPT_KEY]);
        }

        return $guzzleOptions;
    }

    /**
     * effectue une requete
     * @param string $method
     * @param array $pathSegments
     * @param array $options
     * @return ResponseInterface
     * @throws \InvalidArgumentException
     * @throws \UnexpectedValueException
     * @throws \GuzzleHttp\Exception\GuzzleException
     * @throws HttpRequestException
     * @throws \Exception
     */
    public function request($method, array $pathSegments, $options = []) : ResponseInterface
    {
        // récuperation du schema
        $scheme = 'http';
        if ((bool)$this->readOption($options, HttpRequestOptions::SECURE, false) === true) {
            $scheme = 'https';
        }

        // construction du chemin
        $path = implode('/', $pathSegments);
        $uri = $scheme . '://' . $this->requestHost . '/' . trim($path, '/');

        // init tableau options pour guzzle
        $guzzleOptions = $this->convertOptionsForGuzzle($options);

        // exec de la requete
        try {
            $response = $this->client->request($method, $uri, $guzzleOptions);
        } catch (RequestException $e) {
            throw $this->convertGuzzleException($e);
        }

        return $response;
    }

    /**
     * @param RequestInterface $request
     * @param array $options
     * @return ResponseInterface
     * @throws \GuzzleHttp\Exception\GuzzleException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     */
    public function send(RequestInterface $request, $options = []) : ResponseInterface
    {
        $guzzleOptions = $this->convertOptionsForGuzzle($options);
        try {
            $response = $this->client->send($request, $guzzleOptions);
        } catch (RequestException $e) {
            throw $this->convertGuzzleException($e);
        }

        return $response;
    }

    /**
     * effectue une requete asynchrone
     * @param string $method
     * @param array $pathSegments
     * @param array $options
     * @return \GuzzleHttp\Promise\PromiseInterface
     * @throws \InvalidArgumentException
     * @throws \UnexpectedValueException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpServerException
     */
    public function requestAsync($method, array $pathSegments, $options = [])
    {
        // récuperation du schema
        $scheme = 'http';
        if ((bool)$this->readOption($options, HttpRequestOptions::SECURE, false) === true) {
            $scheme = 'https';
        }

        // construction du chemin
        $path = implode('/', $pathSegments);
        $uri = $scheme . '://' . $this->requestHost . '/' . trim($path, '/');

        // init tableau options pour guzzle
        $guzzleOptions = $this->convertOptionsForGuzzle($options);

        // exec de la requete
        try {
            $promise = $this->client->requestAsync($method, $uri, $guzzleOptions);
            $promise = $promise->otherwise(function ($reason) {
                if ($reason instanceof RequestException) {
                    $reason = $this->convertGuzzleException($reason);
                }
                return new RejectedPromise($reason);
            });
        } catch (RequestException $e) {
            throw $this->convertGuzzleException($e);
        }

        return $promise;
    }

    /**
     * @param RequestInterface $request
     * @param array $options
     * @return \GuzzleHttp\Promise\PromiseInterface
     * @throws \InvalidArgumentException
     * @throws \UnexpectedValueException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpServerException
     */
    public function sendAsync(RequestInterface $request, $options = [])
    {
        $guzzleOptions = $this->convertOptionsForGuzzle($options);
        try {
            $response = $this->client->sendAsync($request, $guzzleOptions);
        } catch (RequestException $e) {
            throw $this->convertGuzzleException($e);
        }

        return $response;
    }

    /**
     * @param \GuzzleHttp\Exception\RequestException $error
     * @return \Move\Http\Client\Exception\HttpRequestException
     */
    protected function convertGuzzleException(
        RequestException $error
    ) : Exception\HttpRequestException {
        if ($error instanceof ClientException) {
            return new HttpClientException(
                $error->getRequest(),
                $error->getResponse(),
                $error->getMessage(),
                $error
            );
        }
        if ($error instanceof ServerException) {
            return new HttpServerException(
                $error->getRequest(),
                $error->getResponse(),
                $error->getMessage(),
                $error
            );
        }
        return new HttpRequestException(
            $error->getRequest(),
            $error->getResponse(),
            $error->getMessage(),
            $error
        );
    }
}
