<?php

namespace Cms\Client\Member;

use Cms\Client\AbstractClient;
use Cms\Client\CmsApiIndexIterator;
use Cms\Client\Member\Model\MemberRestfulModel;
use Cms\Model\Member\MemberModel;
use Move\Http\Client\HttpRequestOptions;
use POM\DomainObjectInterface;

/**
 * Class MemberClient
 * @package Cms\Client\Member
 */
class MemberClient extends AbstractClient
{

    /** @var string */
    protected $basePath = 'member';

    /**
     * @param array $dataset
     * @return MemberRestfulModel
     */
    public function handleData($dataset = []) : MemberRestfulModel
    {
        return (new MemberRestfulModel())->populate($dataset, true);
    }

    /**
     * @param array $queryParams
     * @return CmsApiIndexIterator|MemberRestfulModel[]
     * @throws \UnexpectedValueException
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function getIndexWithParams($queryParams)
    {
        $result = $this->loadIndexFromClient($this->basePath, $queryParams);
        return $result;
    }

    /**
     * @param int $page
     * @return MemberRestfulModel[]|CmsApiIndexIterator
     */
    public function getIndex($page = 0)
    {
        $result = $this->getBuilder();
        return $result->getIndex($page);
    }

    /**
     * @param string $searchText
     * @return CmsApiIndexIterator|MemberRestfulModel[]
     */
    public function getIndexBySearch($searchText)
    {
        return $this->getBuilder()
            ->setFilter('search', $searchText)
            ->getIndex(-1);
    }

    /**
     * @param $id
     * @param bool $nocache
     * @return \Cms\Client\Member\Model\MemberRestfulModel|null
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function getItemByImportRef($id, bool $nocache = false)
    {
        $user = $this->fetchItem(null, [
            'import_ref' => $id,
        ], $nocache);
        return $user;
    }

    /**
     * @param $email
     * @param bool $nocache
     * @return \Cms\Client\Member\Model\MemberRestfulModel|null
     * @throws \LogicException
     * @throws \InvalidArgumentException
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     */
    public function getItemByEmail($email, bool $nocache = false)
    {
        $user = $this->fetchItem(null, [
            'email' => $email,
        ], $nocache);
        return $user;
    }

    /**
     * @param $id
     * @return null|MemberRestfulModel
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function getItem($id)
    {
        return $this->fetchItem($id);
    }

    /**
     * @param MemberModel $object
     * @return MemberModel|null
     * @throws \UnexpectedValueException
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function updateItem(MemberModel $object)
    {
        return $this->saveItem($object);
    }

    /**
     * @param MemberModel $object
     * @return bool|MemberModel
     * @throws \UnexpectedValueException
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function createItem(MemberModel $object)
    {
        return $this->saveItem($object);
    }

    /**
     * @param $login
     * @param $password
     * @param $scopeId
     * @return DomainObjectInterface
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function loginMember($login, $password, $scopeId) : DomainObjectInterface
    {
        $memberModel = new MemberModel();
        $result = $this->createObjectFromClient(
            $memberModel,
            'member/login',
            [
                'email' => $login,
                'password' => $password,
                'scope' => $scopeId,
            ]
        );

        return $result;
    }

    /**
     * @param $memberId
     * @param $scopeId
     * @param $password
     * @return bool|CmsApiIndexIterator|DomainObjectInterface
     * @throws \UnexpectedValueException
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function createLoginMember($memberId, $scopeId, $password)
    {
        $memberModel = new MemberRestfulModel();
        $result = $this->createObjectFromClient(
            $memberModel,
            'member/' . $memberId . '/login/' . $scopeId,
            ['password' => $password]
        );

        return $result;
    }

    /**
     * @param $memberId
     * @param $scopeId
     * @param null $password
     * @param null $askReset
     * @return MemberRestfulModel
     * @throws \UnexpectedValueException
     * @throws \RuntimeException
     * @throws \Move\Http\Client\Exception\HttpServerException
     * @throws \Move\Http\Client\Exception\HttpRequestException
     * @throws \Move\Http\Client\Exception\HttpClientException
     * @throws \LogicException
     * @throws \InvalidArgumentException
     */
    public function updateLoginMember($memberId, $scopeId, $password = null, $askReset = null)
    {
        $dataToSend = [];
        if (null !== $password) {
            $dataToSend['password'] = $password;
        }
        if (null !== $askReset) {
            $dataToSend['ask_reset'] = $askReset;
        }

        $options = [
            HttpRequestOptions::BODY_CONTENT => $dataToSend,
        ];
        // lance la requete vers le server
        $jsonDataset = $this->request(
            'PUT',
            ['member', $memberId, 'login', $scopeId],
            $options
        );
        $object = new MemberRestfulModel();
        if (\is_array($jsonDataset) && !empty($jsonDataset)) {
            $object = $this->withJsonDataset($object, $jsonDataset);
        }
        return $object;
    }
}
