<?php


namespace Cms\Search\Address;

use Cms\Search\ElasticQueryBuilderInterface;
use Cms\Search\ElasticQueryBuilderTrait;
use function Cms\Utils\geo_diagonal;
use POM\PredefinedType\GpsLocationPoint;

/**
 * Class AddressSearchQueryBuilder
 * @package Cms\Search\Address
 */
class AddressSearchQueryBuilder implements ElasticQueryBuilderInterface
{
    use ElasticQueryBuilderTrait;

    /** Aggregation par category */
    public const AGG_CATEGORY = 'category';

    /** Aggregation par zone administrative */
    public const AGG_GEO_ADMIN = 'geoAdmin';

    /** Aggretat sur les product avec un attr */
    public const AGG_ATTRIBUTE_VALUE = 'attrValue';

    /** Aggretat sur les product avec un attr */
    public const AGG_ATTRIBUTE_QTE = 'attrQte';

    /** Aggregat de zoom */
    public const AGG_GEO_ZOOM = 'zoom';

    /** Renvoi le viewport */
    public const AGG_GEO_VIEWPORT = 'viewport';


    /**
     * AddressSearchQueryBuilder constructor.
     * @param int $scopeId
     * @param string|null $country
     */
    public function __construct(int $scopeId, string $country = null)
    {
        $this->filters['scope_id'] = $scopeId;
        $this->filters['country'] = $country;
        $this->filters['category'] = null;
        $this->filters['codeParent'] = null;
        $this->filters['gpsLocation'] = [];
        $this->filters['codeConcat'] = null;
        $this->filters['attrRange'] = [];
        $this->filters['additional'] = [];
    }

    /**
     * @param int $categoryId
     * @return $this
     */
    public function setCategory(int $categoryId)
    {
        $this->filters['category'] = $categoryId;
        return $this;
    }

    /**
     * @param string $codeParent
     * @return $this
     */
    public function setAdminCodeParent(string $codeParent = null)
    {
        $this->filters['codeParent'] = $codeParent;
        return $this;
    }

    /**
     * @param array|string $codeConcat
     * @return $this
     */
    public function setAdminCodeConcat($codeConcat = null)
    {
        $this->filters['codeConcat'] = (array)$codeConcat;
        return $this;
    }

    /**
     * @param \POM\PredefinedType\GpsLocationPoint $gpsLocation
     * @param string $distance
     * @return $this
     */
    public function setGpsLocation(GpsLocationPoint $gpsLocation, string $distance)
    {
        $this->filters['gpsLocation'] = [$gpsLocation, $distance];
        return $this;
    }

    /**
     * @param array $filters
     * @param bool $merge
     * @return $this
     */
    public function setAdditionalFilters(array $filters, bool $merge = true)
    {
        if (!$merge) {
            $this->filters['additional'] = $filters;
        } else {
            $this->filters['additional'] = array_merge(
                $this->filters['additional'],
                $filters
            );
        }

        return $this;
    }

    /**
     * @param array $northEast
     * @param array $southWest
     * @return $this
     */
    public function setBoundingBox(array $northEast, array $southWest)
    {
        // filter bounding
        $boundingFilter = [
            'geo_bounding_box' => [
                'geo_location' => [
                    'top_right' => array_reverse($northEast),
                    'bottom_left' => array_reverse($southWest),
                ],
            ],
        ];
        $this->setAdditionalFilters([
            $boundingFilter,
        ]);
        return $this;
    }

    /**
     * @param array $northEast
     * @param array $southWest
     * @param int|null $precision
     * @param bool $addInFilter
     * @return $this
     * @throws \InvalidArgumentException
     */
    public function setBoundingBoxFacet(
        array $northEast,
        array $southWest,
        int &$precision = null,
        bool $addInFilter = true
    ) {
        // distance entre les deux points pour trouver la bonne precision
        $diagonal = geo_diagonal($northEast, $southWest);

        /*
            3 > 156.5km x 156km
            4 > 39.1km x 19.5km
            5 > 4.9km x 4.9km
            6 > 1.2km x 609.4m
        */
        if ($precision === null) {
            $precision = 2;
            if ($diagonal < 50) {
                $precision = null;
            } elseif ($diagonal < 600) {
                $precision = 4;
            } elseif ($diagonal < 1500) {
                $precision = 3;
            }
        }

        // filter bounding
        $boundingFilter = [
            'geo_bounding_box' => [
                'geo_location' => [
                    'top_right' => array_reverse($northEast),
                    'bottom_left' => array_reverse($southWest),
                ],
            ],
        ];

        // finalisation du builder (on prepare les deux pour le zoom automatique si besoin)
        $this->addFacet(
            static::AGG_GEO_ZOOM,
            [$precision],
            [$boundingFilter]
        );
        if ($addInFilter === true) {
            $this->setAdditionalFilters([
                $boundingFilter,
            ]);
        }
        return $this;
    }

    /**
     * @param string $attrName
     * @param int $quantityMin
     * @param int $quantityMax
     * @return $this
     * @throws \InvalidArgumentException
     */
    public function setRangeOnAttribute(string $attrName, int $quantityMin = null, int $quantityMax = null)
    {
        if (!$quantityMax && !$quantityMin) {
            throw new \InvalidArgumentException('any quantity passed');
        }
        if (isset($quantityMax, $quantityMin)
            && $quantityMax <= $quantityMin
        ) {
            throw new \InvalidArgumentException('max/min quantity invalid');
        }
        $this->filters['attrRange'][$attrName] = [$quantityMin, $quantityMax];
        return $this;
    }

    /**
     * @return array
     */
    public function getAggsAsArray() : array
    {
        if (!$this->aggs) {
            return [];
        }

        // init aggs
        $aggsQuery = [];

        // aggregat des categorie
        $this->appendSimpleTermAgg(
            $aggsQuery,
            self::AGG_CATEGORY,
            'category_id',
            100,
            ['_term' => 'asc']
        );

        // aggregat des attribut qte
        if (!empty($this->aggs[self::AGG_ATTRIBUTE_QTE])) {
            $this->aggsFilter[self::AGG_ATTRIBUTE_QTE] = $this->getTermsQuery(
                'attributes_nested.name',
                (array)$this->aggs[self::AGG_ATTRIBUTE_QTE]
            );
        }
        $this->appendSimpleTermAgg(
            $aggsQuery,
            self::AGG_ATTRIBUTE_QTE,
            'attributes_nested.quantity',
            150,
            [],
            'attributes_nested'
        );

        // aggregat des attribut
        if (!empty($this->aggs[self::AGG_ATTRIBUTE_VALUE])) {
            $this->aggsFilter[self::AGG_ATTRIBUTE_VALUE] = $this->getTermsQuery(
                'attributes_nested.name',
                (array)$this->aggs[self::AGG_ATTRIBUTE_VALUE]
            );
        }
        $this->appendSimpleTermAgg(
            $aggsQuery,
            self::AGG_ATTRIBUTE_VALUE,
            'attributes_nested.value.raw',
            150,
            [],
            'attributes_nested'
        );

        // recherche uniquement sur la division geographique
        if (isset($this->aggs[self::AGG_GEO_ADMIN])) {
            $searchQuery = ['match_all' => (object)[]];
            $nestedPath = 'geo_admins_nested';
            if (!empty($this->aggs[self::AGG_GEO_ADMIN])
                && \is_numeric($this->aggs[self::AGG_GEO_ADMIN])
            ) {
                $searchQuery = $this->getTermQuery(
                    $nestedPath . '.admin_level',
                    $this->aggs[self::AGG_GEO_ADMIN]
                );
            } elseif (!empty($this->aggsFilter[self::AGG_GEO_ADMIN])) {
                $searchQuery = $this->aggsFilter[self::AGG_GEO_ADMIN];
            }
            $aggsQuery['geo_admins'] = $this->getAggsNestedQuery(
                $nestedPath,
                $searchQuery,
                [
                    self::AGG_GEO_ADMIN => [
                        'terms' => [
                            'size' => 300,
                            'field' => $nestedPath . '.id',
                        ],
                    ],
                    self::AGG_GEO_ADMIN . '_parent' => [
                        'terms' => [
                            'size' => 300,
                            'field' => $nestedPath . '.parent_id',
                        ],
                    ],
                ]
            );
        }

        // zoom geo
        if (!empty($this->aggs[self::AGG_GEO_ZOOM])) {
            $aggFragment = [
                'geohash_grid' => [
                    'size' => 80,
                    'precision' => $this->aggs[self::AGG_GEO_ZOOM],
                    'field' => 'geo_location',
                ],
                'aggs' => [
                    'centroid_' . self::AGG_GEO_ZOOM => [
                        'geo_centroid' => [
                            'field' => 'geo_location',
                        ],
                    ],
                ],
            ];

            $this->appendAggFragment($aggsQuery, self::AGG_GEO_ZOOM, $aggFragment);
        }


        if (!empty($this->aggs[self::AGG_GEO_VIEWPORT])) {
            $aggFragment = [
                'geo_bounds' => [
                    'field' => 'geo_location',
                ],
            ];

            $this->appendAggFragment($aggsQuery, self::AGG_GEO_VIEWPORT, $aggFragment);
        }

        return $aggsQuery;
    }

    /**
     * @return array
     */
    public function getQueryAsArray() : array
    {
        $query = [];

        if (!empty($this->filters['category'])) {
            $query['bool']['filter'][] = $this->getTermQuery(
                'category_id',
                $this->filters['category']
            );
        }

        if (!empty($this->filters['scope_id'])) {
            $query['bool']['filter'][] = $this->getTermQuery(
                'scope_id',
                $this->filters['scope_id']
            );
        }

        // filtre sur la zone gps
        if ($this->filters['gpsLocation']) {
            [$gpsLocation, $distance] = $this->filters['gpsLocation'];
            if ($gpsLocation instanceof GpsLocationPoint) {
                $subQuery = $this->getBoolQuery([
                    [
                        'geo_distance' => [
                            'distance' => $distance,
                            'geo_location' => [
                                'lat' => $gpsLocation->latitude,
                                'lon' => $gpsLocation->longitude,
                            ],
                        ],
                    ],
                ]);
                $query['bool']['filter'][] = $subQuery;
            }
        } else {
            // filtre sur la zone administrative
            if (($adminCodeConcat = $this->filters['codeConcat']) !== null) {
                $nestedPath = 'geo_admins_nested';
                $subQuery = $this->getBoolQuery([
                    $this->getTermsQuery($nestedPath . '.admin_code_concat', $adminCodeConcat),
                ]);
                $nestedQuery = $this->getNestedQuery($nestedPath, [$subQuery]);
                $query['bool']['filter'][] = $nestedQuery;
            }
            // filtre sur la zone administrative
            if (($adminCodeParent = $this->filters['codeParent']) !== null) {
                $nestedPath = 'geo_admins_nested';
                $subQuery = $this->getBoolQuery([
                    $this->getTermQuery($nestedPath . '.admin_code_parent', $adminCodeParent),
                ]);
                $nestedQuery = $this->getNestedQuery($nestedPath, [$subQuery]);
                $query['bool']['filter'][] = $nestedQuery;
            }
            if (!$adminCodeParent
                && !$adminCodeConcat
                && !empty($this->filters['country'])
            ) {
                $query['bool']['filter'][] = $this->getTermQuery(
                    'country_code',
                    $this->filters['country']
                );
            }
        }

        // ajoute les filtre additionel
        if ($this->filters['additional']) {
            foreach ($this->filters['additional'] as $additionalFilter) {
                $query['bool']['must'][] = $additionalFilter;
            }
        }

        // recherche par prix
        if (!empty($this->filters['attrRange'])
            && \is_array($this->filters['attrRange'])
        ) {
            $subQuery = [];
            foreach ($this->filters['attrRange'] as $attrName => [$min, $max]) {
                $subQuery[] = $this->getBoolQuery([
                    $this->getTermQuery(
                        'attributes_nested.name',
                        $attrName
                    ),
                    $this->getRangeQuery(
                        'attributes_nested.quantity',
                        $max,
                        $min
                    ),
                ]);
            }
            $query['bool']['filter'][] = $this->getNestedQuery(
                'attributes_nested',
                count($subQuery) > 1 ? $this->getBoolQuery($subQuery, 'should') : $subQuery[0]
            );
        }

        return $query ? ['query' => $query] : [];
    }
}
