<?php

namespace Cms\Search\Contract;

use Cms\LinkResource\LinkResourcePrefsIdentifier;
use Cms\Search\ElasticQueryBuilderInterface;
use Cms\Search\ElasticQueryBuilderTrait;
use Elastica\Util;

/**
 * Class ContractSearchQueryBuilder
 * @package Cms\Search\Contract
 */
class ContractSearchQueryBuilder implements ElasticQueryBuilderInterface
{
    use ElasticQueryBuilderTrait;

    /** Aggregat sur les thematics */
    public const AGG_THEMATICS = 'thematicSuggest';

    /**
     * ContractSearchQueryBuilder constructor.
     * @param int $scopeId
     * @param string $state
     */
    public function __construct(
        int $scopeId,
        string $state = null
    ) {
        $this->filters['scopeId'] = $scopeId;
        $this->filters['state'] = $state;
        $this->filters['productProfil'] = null;
        $this->filters['configResource'] = null;
        $this->filters['configResourceSource'] = null;
        $this->filters['advertiserResource'] = null;
        $this->filters['configLanguage'] = null;
        $this->filters['begin_at'] = null;
        $this->filters['end_at'] = null;
    }

    /**
     * @param int $scopeId
     * @return $this
     */
    public function setScopeId(int $scopeId)
    {
        $this->filters['scopeId'] = $scopeId;
        return $this;
    }

    /**
     * @param string|array $productProfil
     * @return $this
     */
    public function setProductProfil($productProfil)
    {
        if (!\is_array($productProfil)) {
            $productProfil = [$productProfil];
        }
        $this->filters['productProfil'] = $productProfil;
        return $this;
    }

    /**
     * @param \DateTime $beginAt
     * @return $this
     */
    public function setBeginAt(\DateTime $beginAt)
    {
        $this->filters['begin_at'] = $beginAt->getTimestamp() * 1000;

        return $this;
    }

    /**
     * @param \DateTime $endAt
     * @return $this
     */
    public function setEndAt(\DateTime $endAt)
    {
        $this->filters['end_at'] = $endAt->getTimestamp() * 1000;

        return $this;
    }

    /**
     * @param string $language
     * @return $this
     */
    public function setConfigLanguage(string $language)
    {
        $this->filters['configLanguage'] = $language;
        return $this;
    }

    /**
     * @param string $type
     * @param mixed|null $ref
     * @return $this
     */
    public function setConfigResource(string $type, $ref = null)
    {
        if ($ref && !\is_array($ref)) {
            $ref = [$ref];
        }
        $this->filters['configResource'][$type] = $ref;
        return $this;
    }

    /**
     * @param array $sourceFiels
     * @return $this
     */
    public function setConfigInnerHitsSource(array $sourceFiels)
    {
        $this->filters['configResourceSource'] = $sourceFiels;
        return $this;
    }

    /**
     * @param string $type
     * @param mixed|null $ref
     * @param int|null $boost
     * @return $this
     */
    public function setAdvertiserResource(string $type, $ref = null, int $boost = null)
    {
        if ($ref && !\is_array($ref)) {
            $ref = [$ref];
        }
        $this->filters['advertiserResource'][$type] = [$ref, $boost];
        return $this;
    }

    /**
     * @param string $state
     * @return $this
     */
    public function setState(string $state = null)
    {
        $this->filters['state'] = $state;
        return $this;
    }

    /**
     * @param string|null $queryString
     * @param string $addressType
     * @return array
     */
    public function getAddressBoolSearchQuery(
        string $queryString = null,
        string $addressType
    ) : array {
        $nestedPath = 'advertiser.addresses_nested';
        $subQuery[] = [
            'term' => [
                $nestedPath . '.type' => $addressType,
            ],
        ];
        if ($queryString !== null && \strlen($queryString) > 0) {
            $subQuery[] = [
                'match' => [
                    $nestedPath . '.city_name_fre' => [
                        'query' => $queryString,
                        'operator' => 'and',
                    ],
                ],
            ];
        }
        return $this->getBoolQuery($subQuery);
    }

    /**
     * @return array
     */
    public function getAggsAsArray() : array
    {
        if (empty($this->aggs)) {
            return [];
        }
        $aggsQuery = [];

        // recherche uniquement sur les thematic
        if (isset($this->aggs[self::AGG_THEMATICS])) {
            $aggFragment = [
                'terms' => [
                    'size' => $this->aggsSize,
                    'field' => 'product_configs.resource_ref',
                    'order' => ['_count' => 'desc'],
                ],
                'aggs' => [
                    'campsites' => [
                        'nested' => [
                            'path' => 'product_configs.config_resources',
                        ],
                        'aggs' => [
                            'campsiteSuggest' => [
                                'terms' => [
                                    'size' => $this->aggsSize * 10,
                                    'field' => 'product_configs.config_resources.resource_ref',
                                    'order' => ['_count' => 'desc'],
                                ],
                            ],
                        ],
                    ],
                ],
            ];
            if (!empty($this->aggsFilter[self::AGG_THEMATICS])) {
                $subAggs[self::AGG_THEMATICS . '_filter'] = [
                    'filter' => $this->aggsFilter[self::AGG_THEMATICS],
                    'aggs' => [
                        self::AGG_THEMATICS => $aggFragment,
                    ],
                ];
            } else {
                $subAggs[self::AGG_THEMATICS] = $aggFragment;
            }
            $aggsQuery['thematics'] = [
                'nested' => [
                    'path' => 'product_configs',
                ],
                'aggs' => $subAggs,
            ];
        }
        return $aggsQuery;
    }

    /**
     * @return array
     */
    public function getQueryAsArray() : array
    {
        // filtre obligatoire
        $queryParams = $this->query_params;
        $queryParams['bool']['filter'][] = [
            'term' => ['scope_id' => $this->filters['scopeId']],
        ];
        if ($this->filters['state'] !== null) {
            $queryParams['bool']['filter'][] = [
                'term' => ['state' => $this->filters['state']],
            ];
        }

        // Filtre sur les dates de début et de fin de validité
        if ($this->filters['begin_at'] && $this->filters['end_at']) {
            $queryParams['bool']['filter'][] = $this->getRangeQuery(
                'begin_at',
                $this->filters['end_at'],
                $this->filters['begin_at'],
                false
            );
            $queryParams['bool']['filter'][] = $this->getRangeQuery(
                'end_at',
                $this->filters['end_at'],
                $this->filters['begin_at'],
                true,
                false
            );
        } elseif ($this->filters['end_at']) {
            $queryParams['bool']['filter'][] = $this->getRangeQuery(
                'end_at',
                $this->filters['end_at']
            );
        } elseif ($this->filters['begin_at']) {
            $queryParams['bool']['filter'][] = $this->getRangeQuery(
                'begin_at',
                null,
                $this->filters['begin_at'],
                false,
                true
            );
        }

        $productProfil = $this->filters['productProfil'];
        $advertiserResource = $this->filters['advertiserResource'];
        $configResource = $this->filters['configResource'];

        // filtre sur les profil de product
        if ($productProfil) {
            $configNestedQuery[] = $this->getTermsQuery(
                'product_configs.profil',
                array_values($productProfil)
            );
        }
        if ($this->filters['configLanguage']) {
            $configNestedQuery[] = $this->getTermQuery(
                'product_configs.language',
                $this->filters['configLanguage']
            );
        }

        // filtre sur les config
        foreach ($configResource ?: [] as $resType => $resVal) {
            $subQuery = [];
            $subQuery[] = $this->getTermQuery(
                'product_configs.resource_type',
                $resType
            );
            if ($resVal !== null) {
                $subQuery[] = $this->getTermsQuery(
                    'product_configs.resource_ref',
                    array_values($resVal)
                );
            }
            $configResQuery[] = $this->getBoolQuery($subQuery);
            unset($subQuery);

            $subQuery = [];
            $subQuery[] = $this->getTermQuery(
                'product_configs.config_resources.resource_type',
                $resType
            );
            if ($resVal !== null) {
                $subQuery[] = $this->getTermsQuery(
                    'product_configs.config_resources.resource_ref',
                    array_values($resVal)
                );
            }
            $configResQuery[] = $this->getNestedQuery(
                'product_configs.config_resources',
                $this->getBoolQuery($subQuery)
            );
            unset($subQuery);
        }
        if (!empty($configResQuery)) {
            $configNestedQuery[] = $this->getBoolQuery($configResQuery, 'should');
        }
        if (!empty($configNestedQuery)) {
            $configNestedQuery[] = $this->getTermQuery(
                'product_configs.state',
                'online'
            );
            $sources = [];
            if ($this->filters['configResourceSource']) {
                $sources = ['_source' => $this->filters['configResourceSource']];
            }
            $queryParams['bool']['filter'][] = $this->getNestedQuery(
                'product_configs',
                $this->getBoolQuery($configNestedQuery),
                array_merge(['size' => 500], $sources)
            );
        }

        // filtre sur l'advertiser
        if ($advertiserResource !== null) {
            $subQueries = ['optional' => [], 'forced' => []];
            foreach ($advertiserResource as $advType => [$advRefs, $boost]) {
                $subQuery = [];
                $subQuery[] = $this->getTermQuery(
                    'advertiser.advertiser_type',
                    $advType
                );
                if ($advRefs) {
                    $subQuery[] = $this->getTermsQuery('advertiser.id', $advRefs);
                }
                if ($boost) {
                    $subQueries['optional'][$boost][] = $this->getBoolQuery($subQuery);
                } else {
                    $subQueries['forced'][] = $this->getBoolQuery($subQuery);
                }
            }
            if ($subQueries['forced']) {
                $queryParams['bool']['filter'][] = $this->getNestedQuery(
                    'advertiser',
                    $this->getBoolQuery($subQueries['forced'], 'should'),
                    ['_source' => ['advertiser.id', 'advertiser.advertiser_type']]
                );
            }
            if ($subQueries['optional']) {
                foreach ($subQueries['optional'] as $boost => $subQueries) {
                    $nestedQuery = $this->getNestedQuery(
                        'advertiser',
                        $this->getBoolQuery($subQueries, 'should'),
                        ['_source' => ['advertiser.id', 'advertiser.advertiser_type']],
                        $boost
                    );
                    $queryParams['bool']['should'][] = $nestedQuery;
                }
            }
        }

        // lecture de la requete
        $queryString = Util::escapeTerm(
            $this->filters['searchText'] ?: ''
        );
        if (!empty($queryString)) {
            // recherche uniquement sur l'address geographique
            $nestedQuery = $this->getNestedQuery(
                'advertiser.addresses_nested',
                [
                    $this->getAddressBoolSearchQuery(
                        $queryString,
                        LinkResourcePrefsIdentifier::ADDRESS_LOC
                    ),
                ]
            );
            $stringQuery[] = $nestedQuery;

            // recherche uniquement sur le label de l'advertiser
            $nestedQuery = $this->getNestedQuery(
                'advertiser',
                [
                    $this->getMatchQuery(
                        'advertiser.advertiser_label',
                        $queryString
                    ),
                ]
            );
            $stringQuery[] = $nestedQuery;

            $boolQuery = $this->getBoolQuery($stringQuery, 'should');
            unset($stringQuery);

            $queryParams['bool']['filter'][] = [
                $boolQuery,
            ];
        }

        return ['query' => $queryParams];
    }
}
