<?php

namespace Move;

/**
 * Class ConfigRegistry
 * @package Move
 */
class ConfigRegistry
{

    /** @var \Move\Config[] */
    private static $configs = [];

    /**
     * @param string $name
     * @return \Move\Config|null
     * @throws \InvalidArgumentException
     */
    public static function getInstance(string $name)
    {
        if (!isset(self::$configs[$name])) {
            throw new \InvalidArgumentException(
                sprintf('Requested "%s" config instance is not in the registry', $name)
            );
        }

        return static::$configs[$name];
    }

    /**
     * @param string $name
     * @param \Move\Config $config
     * @param bool $overwrite
     * @throws \InvalidArgumentException
     */
    public static function addConfig(string $name, Config $config, bool $overwrite = false)
    {
        if (!$overwrite && isset(static::$configs[$name])) {
            throw new \InvalidArgumentException('Config with the given name already exists');
        }

        static::$configs[$name] = $config;
    }

    /**
     * @param string|\Move\Config $config Name or config instance
     */
    public static function removeConfig($config)
    {
        if ($config instanceof Config) {
            if (false !== ($idx = array_search($config, self::$configs, true))) {
                unset(self::$configs[$idx]);
            }
        } else {
            unset(self::$configs[$config]);
        }
    }

    /**
     * Clears the registry
     */
    public static function clear()
    {
        self::$configs = [];
    }
}
