<?php

namespace Move\Command\Command;

use PHPUnit\Framework\Error\Error;
use PHPUnit\Framework\TestCase;

/**
 * Class FetchAllCommandTest
 * @package Move\Command\Command
 */
class FetchAllCommandTest extends TestCase
{

    /**
     * @return array
     */
    public function provideInstall()
    {
        return [
            [
                null, null
            ],
            [
                ['title' => 'DESC'], null
            ],
            [
                null, ['title' => 'aragon']
            ],
            [
                ['title' => 'DESC'], ['title' => 'aragon']
            ]
        ];
    }

    public function provideFiltersValue()
    {
        return [
            ['title', 'argon'],
            ['status', 'online'],
            ['has_value', false],
            ['mod_date', null],
            ['created_at', new \DateTime()],
        ];
    }

    public function provideBadFilters()
    {
        return [
            [false],
            [null],
            ['filtersName'],
        ];
    }

    public function provideFilters()
    {
        return [
            [['title' => 'aragon', 'status' => 'online']],
            [['title' => 'aragon', 'status' => 'offline']],
            [[]],
        ];
    }

    public function provideBadSortCols()
    {
        return [
            [false],
            [null],
            ['filtersName'],
            //[["filtersName" => 'FLOP']],
        ];
    }

    public function provideSortCols()
    {
        return [
            [['title' => 'DESC', 'status' => 'ASC']],
            [['title' => 'ASC']],
            [[]],
        ];
    }

    public function provideSortColsValues()
    {
        return [
            ['title', 'DESC'],
            ['status', 'ASC'],
            ['status', 'asc'],
            ['has_data', 'desc'],
        ];
    }

    /**
     * @dataProvider provideInstall
     * @param array|null $sortCols
     * @param array|null $filters
     */
    public function testConstructCommand($sortCols, $filters)
    {
        $cmd = new FetchAllCommand($sortCols, $filters);
        $this->assertTrue(\is_array($cmd->getFilters()));
        $this->assertTrue(\is_array($cmd->getSortCols()));
    }

    /**
     * @dataProvider provideBadFilters
     * @expectedException Error
     * @param $filters
     */
    public function testWithBadFilters($filters)
    {
        $cmd = new FetchAllCommand();
        $cmd->withFilters($filters);
    }

    /**
     * @dataProvider provideFilters
     * @param $filters
     */
    public function testWithFilters($filters)
    {
        $cmd = new FetchAllCommand();
        $this->assertNotSame($cmd, $filterCmd = $cmd->withFilters($filters));
        $this->assertEquals($filterCmd->getFilters(), $filters);
    }

    /**
     * @dataProvider provideFiltersValue
     * @param $filterName
     * @param $filterValue
     */
    public function testGetFilterValue($filterName, $filterValue)
    {
        $cmd = new FetchAllCommand(null, [$filterName => $filterValue]);
        $this->assertEquals($filterValue, $cmd->getFilterValue($filterName));
    }

    /**
     * @dataProvider provideBadSortCols
     * @expectedException Error
     * @param $sortCols
     */
    public function testWithBadSortCols($sortCols)
    {
        $cmd = new FetchAllCommand();
        $cmd->withSortCols($sortCols);
    }

    /**
     * @dataProvider provideSortCols
     * @param $sortCols
     */
    public function testWithSortCols($sortCols)
    {
        $cmd = new FetchAllCommand();
        $this->assertNotSame($cmd, $sortColCmd = $cmd->withSortCols($sortCols));
        $this->assertEquals($sortColCmd->getSortCols(), $sortCols);
    }

    /**
     * @dataProvider provideSortColsValues
     * @param $sortCol
     * @param $sortOrder
     */
    public function testGetSortOrder($sortCol, $sortOrder)
    {
        $cmd = new FetchAllCommand([$sortCol => $sortOrder]);
        $this->assertEquals(strtoupper($sortOrder), $cmd->getSortOrder($sortCol));
    }

}
