<?php

namespace Move\Jato\Mapper;

use GuzzleHttp\Client;
use Move\Command\Exception\BadParameterException;
use Move\Http\Client\GuzzleClientAdapter;
use Move\Jato\JatoClient;
use Move\Jato\Model\MakeModel;
use PHPUnit\Framework\TestCase;

/**
 * Class MakeMapperTest
 * @package Move\Jato\Mapper
 */
class MakeMapperTest extends TestCase
{
    /**
     * @return array
     */
    public function HandleDataProvider() : array
    {
        $data = <<<'JSONDOC'
[
    {
        "databaseName": "SSCF_CS2002",
        "name": "France",
        "flagUrl": "https://sslphotos.jato.com/Flags/F.png",
        "makes": [
            {
                "makeNameGlobal": "ABARTH",
                "makeNameLocal": "Abarth",
                "makeKey": "Abarth",
                "makeNameToDisplay": "Abarth",
                "logoUrl": "https://sslphotos.jato.com/Manufacturers/logo_abarth.png",
                "minPrice": 15899,
                "maxPrice": 35901
            },
            {
                "makeNameGlobal": "ALFA ROMEO",
                "makeNameLocal": "Alfa Romeo",
                "makeKey": "Alfa Romeo",
                "makeNameToDisplay": "Alfa Romeo",
                "logoUrl": "https://sslphotos.jato.com/Manufacturers/logo_ALFA ROMEO.png",
                "minPrice": 13240,
                "maxPrice": 69999
            },
            {
                "makeNameGlobal": "ASTON MARTIN",
                "makeNameLocal": "Aston Martin",
                "makeKey": "Aston Martin",
                "makeNameToDisplay": "Aston Martin",
                "logoUrl": "https://sslphotos.jato.com/Manufacturers/logo_ASTON MARTIN.png",
                "minPrice": 104038,
                "maxPrice": 191201
            }
        ]
    }
]
JSONDOC;

        $return = json_decode($data, true);
        return [$return];
    }

    /**
     * @return MakeMapper
     */
    public function testConstruct() : MakeMapper
    {
        $email = 'nlegall@movepublishing.com';
        $password = 'king7queen';

        $client = new JatoClient(
            new GuzzleClientAdapter(new Client(), 'webapi-live.jato.com/jato.carspecs.api'), $email,
            $password
        );

        $mapper = new MakeMapper($client);
        $clientTest = $mapper->getClient();

        $this->assertInstanceOf(JatoClient::class, $clientTest, 'Client Error');
        $this->assertEquals($client, $clientTest, 'Client not equal');

        return $mapper;
    }

    /**
     * @dataProvider handleDataProvider
     * @param array $data
     */
    public function testHandleData(array $data)
    {
        $email = 'nlegall@movepublishing.com';
        $password = 'king7queen';

        $client = new JatoClient(
            new GuzzleClientAdapter(new Client(), 'webapi-live.jato.com/jato.carspecs.api'), $email,
            $password
        );
        $mapper = new MakeMapper($client);

        $data = $data['makes'];
        $return = [];
        foreach ($data as $make) {
            $makeModel = $mapper->handleData($make);
            $return[] = $makeModel;
        }
        $this->assertNotEmpty($return, 'Array empty');

        foreach ($return as $makeTest) {
            $this->assertInstanceOf(MakeModel::class, $makeTest, 'Not MakeModel');
        }
    }

    /**
     * @depends testConstruct
     * @param MakeMapper $mapper
     */
    public function testFetchAllMakes(MakeMapper $mapper)
    {
        $dataBase = 'SSCF_CS2002';
        $result = $mapper->fetchAllMakes($dataBase);
        $this->assertInternalType('array', $result, 'FetchAllMakes not an array');
        $this->assertNotEmpty($result, 'FetchAllMakes empty');
        foreach ($result as $make) {
            $this->assertInstanceOf(MakeModel::class, $make, 'Not a MakeModel');
        }
    }

    /**
     * @depends testConstruct
     * @param MakeMapper $mapper
     */
    public function testFetchMake(MakeMapper $mapper)
    {
        $dataBase = 'SSCF_CS2002';
        $makeKey = 'Aston Martin';
        $result = $mapper->fetchMake($dataBase, $makeKey);

        $this->assertInstanceOf(MakeModel::class, $result, 'FetchMake error');
    }

    /**
     * @depends testConstruct
     * @param MakeMapper $mapper
     */
    public function testFetchMakeInvalid(MakeMapper $mapper)
    {
        $this->expectException(BadParameterException::class);

        $dataBase = 'SSCF_CS2002';
        $makeKey = null;
        $mapper->fetchMake($dataBase, $makeKey);
    }

}