<?php


namespace Move\Command;

use Move\Command\Command\FetchAllCommand;
use Move\Command\Exception\BadParameterException;

/**
 * Class SanitizeCommandTrait
 * @package Move\Command
 */
trait SanitizeCommandTrait
{

    /**
     * @param array $cols
     * @param array $colsAuthorized
     * @throws \Move\Command\Exception\BadParameterException
     * @return void
     */
    private function throwErrorOnDiff(array $cols, array $colsAuthorized)
    {
        $colDiffs = array_diff_key($cols, array_flip($colsAuthorized));
        if (!empty($colDiffs)) {
            throw new BadParameterException(
                'cols are not correct : [' .
                implode(', ', array_keys($colDiffs)) . '], please use : [' .
                implode(', ', $colsAuthorized) . ']'
            );
        }
    }

    /**
     * @param FetchAllCommand $command
     * @param array $colsAuthorized
     * @param bool $slient
     * @return array
     * @throws \Move\Command\Exception\BadParameterException
     */
    protected function sanitizeSortCols(
        FetchAllCommand $command,
        array $colsAuthorized,
        $slient = true
    ) : array {
        $sortCols = $command->getSortCols();
        // throw exception if not valid sort cols
        if ($slient === false) {
            $this->throwErrorOnDiff($sortCols, $colsAuthorized);
        }
        // filtre sur les sortCols
        $sortCols = array_intersect_key($sortCols, array_flip($colsAuthorized));
        // check des valeurs de tri
        return array_map('strtoupper', array_filter($sortCols, function ($sortOrder) {
            return \in_array(strtoupper($sortOrder), ['DESC', 'ASC']);
        }));
    }

    /**
     * @param FetchAllCommand $command
     * @param array $colsAuthorized
     * @param bool $slient
     * @return array
     * @throws \Move\Command\Exception\BadParameterException
     */
    protected function sanitizeFilters(
        FetchAllCommand $command,
        array $colsAuthorized,
        $slient = true
    ) : array {
        $filters = $command->getFilters();
        // throw exception if not valid filter cols
        if ($slient === false) {
            $this->throwErrorOnDiff($filters, $colsAuthorized);
        }
        // fitre des filtre avec les nom de colonne
        return array_intersect_key($filters, array_flip($colsAuthorized));
    }

    /**
     * @param \Move\Command\Command\FetchAllCommand $command
     * @param array $colsRequired
     * @param bool $acceptEmpty
     * @return array
     * @throws \Move\Command\Exception\BadParameterException
     */
    protected function checkRequired(
        FetchAllCommand $command,
        array $colsRequired,
        bool $acceptEmpty = false
    ) : array {
        $filters = $command->getFilters();
        // fitre des filtre avec les nom de colonne
        $filterRequires = array_intersect_key($filters, array_flip($colsRequired));
        if (!$acceptEmpty) {
            $filterRequires = array_filter($filterRequires);
        }
        if (count($filterRequires) !== count($colsRequired)) {
            throw new BadParameterException(
                'filter required are missing, please use : [' .
                implode(', ', $colsRequired) . ']'
            );
        }
        return $filters;
    }
}
