<?php

namespace Move\Http\Strategy\ExceptionHandler;

use League\Route\Http\Exception\BadRequestException;
use League\Route\Http\Exception\ConflictException;
use League\Route\Http\Exception\HttpExceptionInterface;
use Move\Environment;
use Move\Http\Strategy\ResponseFactory;

/**
 * Class ExceptionHandler
 * @package Move\Http\Strategy\ExceptionHandler
 */
class ExceptionHandler implements ExceptionHandlerInterface
{
    /**
     * @var bool
     */
    private $withTrace;

    /**
     * @var bool
     */
    private $withPrevious;

    /**
     * ExceptionHandler constructor.
     * @param bool $withTrace
     * @param bool $withPrevious
     */
    public function __construct(bool $withTrace = null, bool $withPrevious = null)
    {
        if ($withTrace === null) {
            $withTrace = (new Environment())->isDebug();
        }
        if ($withPrevious === null) {
            $withPrevious = (new Environment())->isDebug();
        }
        $this->withTrace = $withTrace;
        $this->withPrevious = $withPrevious;
    }

    /**
     * @param \Exception $exception
     * @return ResponseFactory
     */
    public function handle(\Exception $exception)
    {
        $httpException = $exception;
        // gestion spécifique PDO
        if ($exception instanceof \PDOException) {
            $httpException = $this->convertPDOException($exception);
        }

        $dataset = [
            'message' => $exception->getMessage(),
            'code' => $exception->getCode(),
        ];

        // message stack
        if ($this->withPrevious) {
            $previous = $exception;
            do {
                $messages[] = sprintf(
                    "%s:%d %s (%d) [%s]\n",
                    $previous->getFile(),
                    $previous->getLine(),
                    $previous->getMessage(),
                    $previous->getCode(),
                    \get_class($previous)
                );
            } while ($previous = $previous->getPrevious());
            $dataset['messageStack'] = $messages;
        }

        if ($this->withTrace === true) {
            $dataset['trace'] = $exception->getTrace();
            $dataset['file'] = $exception->getFile() . ' (' . $exception->getLine() . ')';
        }

        // gestion spec HTTP
        $headers = [];
        $errorCode = 500;
        if ($httpException instanceof HttpExceptionInterface) {
            $errorCode = $httpException->getStatusCode();
            $headers = $httpException->getHeaders();
        }

        // fabrique de la resource error
        $dataset['code'] = $errorCode;
        return new ResponseFactory($dataset, $errorCode, $headers);
    }

    /**
     * @param \PDOException $exception
     * @return HttpExceptionInterface|\PDOException
     */
    public function convertPDOException(\PDOException $exception)
    {
        if (!empty($exception->errorInfo[1])) {
            $message = $exception->errorInfo[2];
            switch ($exception->errorInfo[1]) {
                case 1062:
                    // conflict
                    return new ConflictException($message, $exception);
                    break;
                case 1452:
                    // erreur sur les clé etrangere
                    return new BadRequestException($message, $exception);
                    break;
            }
        }
        return $exception;
    }
}
