<?php


namespace Cms\Client\AddressBook\Transformer;

use Cms\Client\AddressBook\Model\TripRestfulModel;
use Cms\Client\Geodata\Helper\GeodataTransformerTrait;
use Cms\Client\Service\GeodataService;
use Cms\Client\Service\GeodataServiceAwareTrait;
use Cms\Model\AddressBook\Trip\TripStepModel;
use Cms\Model\AddressBook\Trip\TripStepType;
use emcconville\Polyline\GoogleTrait;
use Move\ObjectMapper\ObjectTransformer;
use POM\DomainObjectInterface;

/**
 * Class TripTransformer
 * @package Cms\Client\AddressBook\Transformer
 */
class TripTransformer extends ObjectTransformer
{
    use GeodataServiceAwareTrait;
    use GeodataTransformerTrait;
    use GoogleTrait;

    /**
     * AddressTransformer constructor.
     * @param GeodataService $geodataService
     */
    public function __construct(GeodataService $geodataService)
    {
        $this->setGeodataService($geodataService);
    }

    /**
     * @param TripRestfulModel|DomainObjectInterface $tripRestfulModel
     * @return array
     */
    public function transform(DomainObjectInterface $tripRestfulModel)
    {
        $result = parent::transform($tripRestfulModel);
        if (!$tripRestfulModel instanceof TripRestfulModel) {
            return $result;
        }

        // ajout de donnée d'attribut par clé/val
        $result['attrs'] = [];
        if (!empty($result['attributes'])) {
            foreach ($result['attributes'] as $attribute) {
                $result['attrs'][$attribute['name']] = $attribute['value'];
            }
        }

        // has gallery
        if ($tripRestfulModel->assets) {
            $result['hasGallery'] = true;
        }

        // gestion de l'emplacement du trip
        if (!empty($tripRestfulModel->steps)) {
            $startStep = null;
            $points = [];
            foreach ($tripRestfulModel->steps as $step) {
                if ((string)$step->type === TripStepType::STEP) {
                    if (empty($startStep)) {
                        $startStep = $step;
                    }
                    $points[] = [$step->gps_location->latitude, $step->gps_location->longitude];
                }
            }
            $encodedPolyline = $this->encodePoints($points);
            $result['encodedPolyline'] = $encodedPolyline;

            // ajoute les infos administrative
            if (isset($startStep) && $startStep instanceof TripStepModel) {
                $result['start_location'] = (new ObjectTransformer())->transform($startStep);
                $result['start_location']['admin2'] = $this->transformAdminModel($startStep);
                if (empty($result['start_location']['admin2'])) {
                    try {
                        $countryModel = $this->getGeodataService()
                            ->getCountryClient()->fetchItem(strtolower($startStep->country_alpha2));
                        if ($countryModel) {
                            $result['start_location']['country'] = (new ObjectTransformer())->transform($countryModel);
                        }
                    } catch (\Exception $e) {
                    }
                } else {
                    $result['start_location']['country'] = $result['start_location']['admin2']['country'];
                }
            }
        }
        return $result;
    }
}
