<?php


namespace Cms\Client\Command\Article;

use Cms\Cache\CacheHelperTrait;
use Cms\Client\Article\Model\CategoryRestfulModel;
use Cms\Client\Command\Article\Command\FetchAllArticleCategoryByFilterCommand;
use Cms\Client\Command\Article\Command\FetchArticleCategoryCommand;
use Cms\Client\Command\CommandExceptionHandleTrait;
use Cms\Client\Service\ArticleService;
use Cms\Model\Article\Category\CategoryState;
use Move\Cache\CacheAwareInterface;
use Move\Command\Exception\BadParameterException;
use Move\Command\Exception\NotFoundException;
use Move\Http\Client\Exception\HttpRequestException;
use Psr\Cache\CacheItemInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;

/**
 * Class RubriqueCommandHandler
 * @package AutoMag\Command
 */
class ArticleCategoryCommandHandler implements LoggerAwareInterface, CacheAwareInterface
{
    use CacheHelperTrait;
    use LoggerAwareTrait;
    use CommandExceptionHandleTrait;

    /**
     * @var ArticleService
     */
    private $articleService;


    /**
     * RubriqueCommandHandler constructor.
     * @param ArticleService $articleService
     */
    public function __construct(
        ArticleService $articleService
    ) {
        $this->articleService = $articleService;
    }

    /**
     * @param FetchAllArticleCategoryByFilterCommand $command
     * @return \Iterator|LinkResourceModel[]
     */
    public function handleFetchAllArticleCategoryByFilter(
        FetchAllArticleCategoryByFilterCommand $command
    ) {
        // creation du builder
        $categoryClient = $this->articleService->getCategoryClient();
        $queryBuilder = $categoryClient->getBuilder();

        // ajoute des filtre
        $parentId = $command->getParentId();
        if ($parentId) {
            $queryBuilder->setParentId($parentId);
        }
        $profil = $command->getProfil();
        if ($profil) {
            $queryBuilder->setProfil($profil);
        }
        $scope = $command->getScopeId();
        if ($scope) {
            $queryBuilder->setScope($scope);
        }

        try {
            // execution de la requete
            $categories = $queryBuilder->getIndex(0);
        } catch (HttpRequestException $e) {
            $this->logHttpRequestException('RubriqueCmdHd_FetchAllRubriqueByFilter', $e);
        } finally {
            if (!isset($categories)) {
                $categories = null;
            }
        }
        return $categories;
    }

    /**
     * @param FetchArticleCategoryCommand $command
     * @return CategoryRestfulModel
     * @throws \LogicException
     * @throws \RuntimeException
     * @throws \Move\Command\Exception\BadParameterException
     * @throws \Move\Command\Exception\NotFoundException
     * @throws \InvalidArgumentException
     */
    public function handleFetchArticleCategory(FetchArticleCategoryCommand $command) : CategoryRestfulModel
    {
        $categoryID = $command->getId();
        $categorySlug = $command->getSlug();
        $scopeId = $command->getScopeId();
        $language = $command->getLanguage();
        $slevel = $command->getSlugLevel();

        $cacheKey = 'article_cat^' . $scopeId . '^' . ($categorySlug ?: $categoryID) . '^' . $language;
        if ($categorySlug && $slevel !== -1) {
            $cacheKey .= '^' . $slevel;
        }

        if (true === $this->hitFromCache($cacheKey, 600, $cacheItem)
            && $cacheItem instanceof CacheItemInterface
        ) {
            $categoryModel = $cacheItem->get();
        } else {
            $categoryModel = null;
            try {
                $categoryClient = $this->articleService->getCategoryClient();
                if ($categoryID) {
                    $categoryModel = $categoryClient->fetchItem($categoryID);
                } elseif ($categorySlug && $scopeId) {
                    $categoryModel = $categoryClient->fetchItemBySlug(
                        $categorySlug,
                        $scopeId,
                        $slevel,
                        $language
                    );
                } else {
                    throw new BadParameterException('slug or id must be defined');
                }
            } catch (HttpRequestException $e) {
                $this->logHttpRequestException('RubriqueCmdHd_FetchRubrique', $e);
            }

            // prise en charge du cache
            $this->setInCache($cacheItem, $categoryModel);
        }

        if (!$categoryModel instanceof CategoryRestfulModel
            || (string)$categoryModel->state !== CategoryState::ONLINE
        ) {
            throw new NotFoundException('id:' . ($categorySlug ?: $categoryID));
        }
        return $categoryModel;
    }
}
