<?php

namespace Move;

use ArrayAccess;

/**
 * Class Config
 * @package Move
 */
class Config implements ArrayAccess
{

    /**
     * @var array
     */
    private $configData;

    /**
     * Config constructor.
     * @param array $configData
     */
    public function __construct(array $configData)
    {
        $this->configData = $configData;
    }

    /**
     * @inheritdoc
     */
    public function offsetExists($offset)
    {
        try {
            $this->offsetGet($offset);
            return true;
        } catch (\OutOfBoundsException $e) {
        }
        return false;
    }

    /**
     * @inheritdoc
     */
    public function offsetGet($offset)
    {
        $configData = $this->configData;
        $indexPart = explode('.', $offset);
        if (count($indexPart) !== 1) {
            $nbIteration = count($indexPart);
            for ($i = 0; $i < $nbIteration; $i++) {
                if (!isset($configData[$indexPart[$i]])) {
                    throw new \OutOfBoundsException("l'offset $offset n'existe pas");
                }
                $configData = $configData[$indexPart[$i]];
            }

            $configData = $this->parseValue($configData);
            return $configData;
        }

        $configData = $this->parseValue($this->configData[$offset]);
        return $configData;
    }

    /**
     * @param mixed $value
     * @return mixed
     */
    protected function parseValue($value)
    {
        if (\is_array($value)) {
            $value = new Config($value);
        } elseif (\is_string($value) && preg_match_all('@urn:([^:]+):([a-zA-Z_0-9]+)@', $value, $match)) {
            foreach ($match[0] as $key => $urnValue) {
                $value = preg_replace(
                    '@' . preg_quote($urnValue, '@') . '@',
                    $this->parseUrn($match[1][$key], $match[2][$key]),
                    $value
                );
            }
        }
        return $value;
    }

    /**
     * @param string $urnName
     * @param string $urnKey
     * @return null
     */
    public function parseUrn($urnName, $urnKey)
    {
        switch ($urnName) {
            case 'env':
                return getenv($urnKey);
                break;
        }
        return null;
    }

    /**
     * @inheritdoc
     */
    public function offsetSet($offset, $value)
    {
        throw new \RuntimeException('Config is readonly');
    }

    /**
     * @inheritdoc
     */
    public function offsetUnset($offset)
    {
        throw new \RuntimeException('Config is readonly');
    }

    /**
     * @return array
     */
    public function getArrayCopy()
    {
        $configData = $this->configData;
        foreach ($configData as $key => &$value) {
            $value = $this->offsetGet($key);
            if ($value instanceof Config) {
                $value = $value->getArrayCopy();
            }
        }
        return $configData;
    }
}
