<?php

namespace Move\Http\Client;

use GuzzleHttp\Client;
use GuzzleHttp\Promise\PromiseInterface;
use GuzzleHttp\Psr7\Response;
use PHPUnit\Framework\TestCase;
use Psr\Http\Message\RequestInterface;

/**
 * Class GuzzleClientAdapterTest
 * @package Move\Http\Client
 */
class GuzzleClientAdapterTest extends TestCase
{
    /**
     * @covers GuzzleClientAdapter::__construct
     * @return GuzzleClientAdapter
     */
    public function testConstruct()
    {
        /** @var Client $client */
        $client = $this->getMockBuilder(Client::class)
            ->getMock();
        $instance = new GuzzleClientAdapter(
            $client,
            'http://localhost'
        );

        $this->assertInstanceOf(
            GuzzleClientAdapter::class,
            $instance
        );
        return $instance;
    }

    /**
     * @depends testConstruct
     * @covers GuzzleClientAdapter::request
     */
    public function testRequest()
    {
        $expected = new Response();

        /** @var Client|\PHPUnit_Framework_MockObject_MockObject $client */
        $client = $this->getMockBuilder(Client::class)
            ->setMethods(['request'])
            ->getMock();

        $client
            ->expects($this->any())
            ->method('request')
            ->willReturn($expected);

        $adapter = new GuzzleClientAdapter(
            $client,
            'localhost'
        );

        $response = $adapter->request('GET', ['test']);

        $this->assertEquals($expected, $response);

        return $adapter;
    }

    /**
     * @depends testConstruct
     * @expectedException \Move\Http\Client\Exception\HttpRequestException
     * @covers GuzzleClientAdapter::request
     */
    public function testRequestThrowsHttpRequestException()
    {
        $client = new Client();
        $adapter = new GuzzleClientAdapter(
            $client,
            'localhost'
        );

        $adapter->request('GET', ['test']);
    }

    /**
     * @depends testConstruct
     * @expectedException \InvalidArgumentException
     * @covers GuzzleClientAdapter::request
     */
    public function testRequestThrowsInvalidArgumentException()
    {
        $client = new Client();
        $adapter = new GuzzleClientAdapter(
            $client,
            'localhost'
        );

        $adapter->request('GET', ['test'], ['tesetOption']);
    }

    /**
     * @param GuzzleClientAdapter $adapter
     * @depends testRequest
     */
    public function testRequestAssync(GuzzleClientAdapter $adapter)
    {
        $promise = $adapter->requestAsync('GET', ['test']);
        $this->assertInstanceOf(PromiseInterface::class, $promise);
    }

    /**
     * @depends testConstruct
     * @covers GuzzleClientAdapter::send
     */
    public function testSend()
    {
        /** @var RequestInterface $request */
        $request = $this->getMockBuilder(RequestInterface::class)
            ->getMock();

        $expectedResponse = new Response();

        /** @var Client|\PHPUnit_Framework_MockObject_MockObject $client */
        $client = $this->getMockBuilder(Client::class)
            ->setMethods(['send'])
            ->getMock();

        $client
            ->expects($this->any())
            ->method('send')
            ->willReturn($expectedResponse);

        $adapter = new GuzzleClientAdapter(
            $client,
            'localhost'
        );

        $response = $adapter->send($request);

        $this->assertEquals($expectedResponse, $response);

        return [
            'adapter' => $adapter,
            'request' => $request
        ];
    }

    /**
     * @depends testSend
     * @param array [adapter, request] $datas
     * @expectedException \InvalidArgumentException
     * @covers GuzzleClientAdapter::request
     */
    public function testSendThrowsInvalidArgumentException(array $datas)
    {
        $adapter = $datas['adapter'];
        $request = $datas['request'];
        $adapter->send($request, ['dummyParams']);
    }

    /**
     * @depends testConstruct
     * @param array [adapter, request] $datas
     */
    public function testSendAssync()
    {
        /** @var RequestInterface $request */
        $request = $this->getMockBuilder(RequestInterface::class)
            ->getMock();

        /** @var RequestInterface $request */
        $expectedPromise = $this->getMockBuilder(PromiseInterface::class)
            ->getMock();

        /** @var Client|\PHPUnit_Framework_MockObject_MockObject $client */
        $client = $this->getMockBuilder(Client::class)
            ->setMethods(['sendAsync'])
            ->getMock();

        $client
            ->expects($this->any())
            ->method('sendAsync')
            ->willReturn($expectedPromise);

        $adapter = new GuzzleClientAdapter(
            $client,
            'localhost'
        );

        $promise = $adapter->sendAsync($request);

        $this->assertInstanceOf(PromiseInterface::class, $promise);
        $this->assertSame($expectedPromise, $promise);

    }
}
