<?php


namespace Cms\Model\Geodata;

use Cms\Model\GpsLocationPointModelTrait;
use Cms\Model\StampAwareModelTrait;
use POM\DomainObjectAbstract;
use POM\PredefinedType\GpsLocationPoint;

/**
 * Class CityModel
 * @package Cms\Model\Geodata
 * @property int $id
 * @property string $country_alpha2
 * @property string $name
 * @property string $name_slug
 * @property string $name_ascii
 * @property string $gps_latitude
 * @property string $gps_longitude
 * @property string $admin1_code
 * @property string $admin2_code
 * @property string $insee_code
 * @property string $population
 * @property string $altitude
 * @property string $superficie
 * @property string $timezone
 * @property int $geonames_id
 * @property \DateTime $geonames_modified_at
 * @property string $osm_id
 * @property \POM\PredefinedType\GpsLocationPoint $gps_location
 * @property int $move_id
 * @property string $carte_amenages
 * @property string $carte_europesud
 * @property string $carte_cc1
 * @property string $carte_cc2
 * @property bool $village_neige
 * @property bool $station_verte
 * @property bool $pavillon_bleu
 */
class CityModel extends DomainObjectAbstract
{
    use StampAwareModelTrait;
    use GpsLocationPointModelTrait {
        GpsLocationPointModelTrait::propertySet as gpsOffsetSet;
    }

    /** @var  int */
    protected $id;

    /** @var  string */
    protected $country_alpha2;

    /** @var  string */
    protected $name;

    /** @var  string */
    protected $name_slug;

    /** @var  string */
    protected $name_ascii;

    /** @var  string */
    protected $gps_latitude;

    /** @var  string */
    protected $gps_longitude;

    /** @var  string */
    protected $admin1_code;

    /** @var  string */
    protected $admin2_code;

    /** @var  string */
    protected $insee_code;

    /** @var  string */
    protected $population;

    /** @var  string */
    protected $altitude;

    /** @var  string */
    protected $superficie;

    /** @var  string */
    protected $timezone;

    /** @var  int */
    protected $geonames_id;

    /** @var  \DateTime */
    protected $geonames_modified_at;

    /** @var  string */
    protected $osm_id;

    /** @var  \POM\PredefinedType\GpsLocationPoint */
    protected $gps_location;

    /** @var  int */
    protected $move_id;

    /** @var  string */
    protected $carte_amenages;

    /** @var  string */
    protected $carte_europesud;

    /** @var  string */
    protected $carte_cc1;

    /** @var  string */
    protected $carte_cc2;

    /** @var  bool */
    protected $village_neige;

    /** @var  bool */
    protected $station_verte;

    /** @var  bool */
    protected $pavillon_bleu;

    /**
     * @param int $level
     * @return string|null
     * @throws \InvalidArgumentException
     */
    public function getAdminCodeSimple(int $level)
    {
        if ($level < 1 || $level > 2) {
            throw new \InvalidArgumentException('level is invalid');
        }
        $codeSplit = explode('.', $this->{'admin' . $level . '_code'});
        return $codeSplit ? array_pop($codeSplit) : null;
    }

    /**
     * @inheritdoc
     */
    public function propertyGet($name)
    {
        if ($name === 'carte_code') {
            $name = 'carte_amenages';
        }
        $result = parent::propertyGet($name);
        if ($result instanceof GpsLocationPoint
            && (!$result->latitude || !$result->longitude)) {
            $result = null;
        }
        return $result;
    }

    /**
     * @param string $offset
     * @param mixed $value
     */
    public function propertySet($offset, $value)
    {
        if ($offset === 'carte_code') {
            $offset = 'carte_amenages';
        }
        if ($offset === 'gps_location') {
            if (is_scalar($value)) {
                $value = trim($value);
            }
            if (\is_array($value) && isset($value['latitude'], $value['longitude'])) {
                $value = (new GpsLocationPoint())->populate([
                    'latitude' => str_replace(',', '.', $value['latitude']),
                    'longitude' => str_replace(',', '.', $value['longitude']),
                ], true);
            }
            if ($value instanceof GpsLocationPoint) {
                parent::propertySet($offset, $value);
            } elseif (!empty($value)) {
                $this->gpsOffsetSet($offset, $value);
            } else {
                parent::propertySet($offset, null);
            }
        } else {
            if ($offset === 'gps_latitude' || $offset === 'gps_longitude') {
                $value = (string)str_replace(',', '.', $value);
            }
            parent::propertySet($offset, $value);
        }
    }
}
