<?php

namespace Cms\Client;

use Traversable;

/**
 * Class CmsApiIndexIterator
 * @package Cms\Client
 */
class CmsApiIndexIterator extends \IteratorIterator implements \Serializable, \Countable
{

    /** @var int */
    private $offset;

    /** @var int */
    private $total;

    /** @var int */
    private $byPage;

    /**
     * @param int $byPage
     * @return \Cms\Client\CmsApiIndexIterator
     */
    public static function createEmpty(int $byPage = 20) : CmsApiIndexIterator
    {
        return new self(new \ArrayIterator(), 0, 0, $byPage);
    }

    /**
     * CmsApiIndexIterator constructor.
     * @param Traversable $iterator
     * @param int $offset
     * @param int $total
     * @param int $byPage
     */
    public function __construct(Traversable $iterator, $offset, $total, $byPage)
    {
        parent::__construct($iterator);
        $this->offset = $offset;
        $this->total = $total;
        $this->byPage = $byPage;
    }

    /**
     * @return mixed|null
     */
    public function getFirst()
    {
        $this->rewind();
        if ($this->valid()) {
            return $this->current();
        }
        return null;
    }

    /**
     * @return int
     */
    public function getOffset()
    {
        return $this->offset;
    }

    /**
     * @return int
     */
    public function getTotal()
    {
        return $this->total;
    }

    /**
     * @return int
     */
    public function getByPage()
    {
        return $this->byPage;
    }

    /**
     * String representation of object
     * @link http://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     * @since 5.1.0
     */
    public function serialize()
    {
        return serialize([
            'iterator' => iterator_to_array($this->getInnerIterator()),
            'total' => $this->total,
            'offset' => $this->offset,
            'byPage' => $this->byPage,
        ]);
    }

    /**
     * Constructs the object
     * @link http://php.net/manual/en/serializable.unserialize.php
     * @param string $serialized <p>
     * The string representation of the object.
     * </p>
     * @return void
     * @since 5.1.0
     */
    public function unserialize($serialized)
    {
        $raw = unserialize($serialized);

        parent::__construct(new \ArrayIterator($raw['iterator']));

        $this->total = $raw['total'];
        $this->offset = $raw['offset'];
        $this->byPage = $raw['byPage'];
    }

    /**
     * Count elements of an object
     * @link http://php.net/manual/en/countable.count.php
     * @return int The custom count as an integer.
     * </p>
     * <p>
     * The return value is cast to an integer.
     * @since 5.1.0
     */
    public function count()
    {
        return $this->getTotal();
    }
}
