<?php

namespace Cms\LinkResource\Provider\Jato;

use Cms\Cache\CacheHelperTrait;
use Cms\LinkResource\LinkResourceType;
use Cms\LinkResource\Provider\AbstractProviderAdapter;
use Cms\LinkResource\Provider\DataAdapterAwareInterface;
use Cms\LinkResource\Provider\SearchAdapterAwareInterface;
use Cms\LinkResource\Provider\SearchAdapterAwareTrait;
use Move\Cache\CacheAwareInterface;
use Move\Jato\JatoClient;
use Move\Jato\JatoMakeService;
use Move\Jato\JatoModelService;
use Move\Jato\Mapper\MakeMapper;
use Move\Jato\Mapper\ModelMapper;
use Move\Jato\Model\MakeModel;
use Move\Jato\Model\ModelModel;
use Move\Jato\Model\VehicleModel;
use Move\Utils\Str;
use Psr\Cache\CacheItemInterface;

/**
 * Class JatoProvider
 * @package Cms\LinkResource\Provider\Jato
 */
class JatoProvider extends AbstractProviderAdapter implements
    SearchAdapterAwareInterface,
    DataAdapterAwareInterface,
    CacheAwareInterface
{
    use CacheHelperTrait;
    use SearchAdapterAwareTrait;

    /* Identifier */
    public const IDENTIFIER = 'jato';

    /* sépratateur des clé */
    public const KEY_SEPARATOR = ':';

    /** @var array */
    protected $types = [
        LinkResourceType::MARQUE,
        LinkResourceType::MODEL,
    ];

    /** @var string */
    protected $identifier = self::IDENTIFIER;

    /** @var \Move\Jato\JatoClient */
    private $client;

    /** @var string */
    private $databaseName;

    /**
     * CmsProvider constructor.
     * @param \Move\Jato\JatoClient $client
     * @param string $databaseName
     */
    public function __construct(JatoClient $client, $databaseName)
    {
        $this->client = $client;
        $this->databaseName = $databaseName;
    }

    /**
     * @param LinkResourceType|string $typeCode
     * @param int $objectId
     * @return string
     */
    public function getLabel($typeCode, $objectId)
    {
        switch ($typeCode) {
            case LinkResourceType::MARQUE:
                $make = $this->getData($typeCode, $objectId);
                if (!$make) {
                    return null;
                }
                return $make->makeNameToDisplay;
                break;
            case LinkResourceType::MODEL:
                $model = $this->getData($typeCode, $objectId);
                if (!$model) {
                    return null;
                }
                return $model->modelNameToDisplay . ' (' . $model->makeNameToDisplay . ')';
                break;
        }
        return null;
    }

    /**
     * @param LinkResourceType|string $typeCode
     * @param int $objectId
     * @return MakeModel|ModelModel|VehicleModel
     */
    public function getData($typeCode, $objectId)
    {
        $cacheKey = 'jatoProviderData^' . $typeCode . '^'
            . $this->databaseName . '^' . Str::toAscii($objectId);
        if ($this->hitFromCache($cacheKey, 3600 * 24, $cacheItem)
            && $cacheItem instanceof CacheItemInterface
        ) {
            return $cacheItem->get();
        }
        $return = null;
        switch ($typeCode) {
            case LinkResourceType::MARQUE:
                $mapper = new MakeMapper($this->client);
                $return = $mapper->fetchMake($this->databaseName, $objectId);
                break;
            case LinkResourceType::MODEL:
                $mapper = new ModelMapper($this->client);
                [$makeKey, $modelKey] = explode(self::KEY_SEPARATOR, $objectId);
                if ($makeKey && $modelKey) {
                    $model = $mapper->fetchAllModels($this->databaseName, $makeKey, $modelKey);
                    $return = $model[0] ?: null;
                }
        }
        $this->setInCache($cacheItem, $return);
        return $return;
    }

    /**
     * @param LinkResourceType|string $typeCode
     * @param mixed $query
     * @return string[]
     * @throws \RuntimeException
     * @throws \Psr\Cache\InvalidArgumentException
     */
    public function execSearch($typeCode, $query)
    {
        $return = [];
        switch ($typeCode) {
            case LinkResourceType::MARQUE:
                $makeMapper = new MakeMapper($this->client);
                $service = new JatoMakeService($makeMapper);
                $service->setItemCachePool($this->getItemCachePool());
                $make = $service->searchBestMake(
                    $this->databaseName,
                    $query
                );
                $return = [$make->makeKey];
                break;
            case LinkResourceType::MODEL:
                $mapper = new ModelMapper($this->client);
                $service = new JatoModelService($mapper);
                $service->setItemCachePool($this->getItemCachePool());
                $models = $service->fetchAllForMake($this->databaseName, $query);
                $return = array_map(function (ModelModel $model) {
                    return $model->makeKey . self::KEY_SEPARATOR . $model->modelKey;
                }, $models);
                break;
        }
        return $return;
    }
}
