<?php


namespace Cms\Restful;

use League\Route\Http\Exception\BadRequestException;
use League\Route\Http\Exception\ConflictException;
use League\Route\Http\Exception\NotFoundException;
use Move\Command\Command\CreateCommand;
use Move\Command\Command\UpdateCommand;
use Move\Command\CommandBusInterface;
use Move\Command\Exception\BadParameterException;
use Move\Command\Exception\OutOfBoundsException;
use Move\Command\Exception\RelationException;
use Move\Filter\Validator\StateValidatorException;
use Move\Utils\Str;

/**
 * Class CommandControllerTrait
 * @package Cms\Restful
 */
trait CommandControllerTrait
{

    /**
     * @return CommandBusInterface
     */
    abstract protected function getCommandBus();

    /**
     * @param object $command
     * @return mixed
     * @throws BadRequestException
     * @throws ConflictException
     * @throws NotFoundException
     */
    protected function execCommand($command)
    {
        $errorMsg = 'Command failed to execute due to exception '
            . "'%s' with message '%s' in %s:%s\nTrace:\n%s";
        try {
            $result = $this->getCommandBus()->handle($command);
        } catch (StateValidatorException $e) {
            $propName = $e->getPropName();
            $errorMsg = 'Command failed to execute due to validation error '
                . "on property '%s' with value %s.\nTrace:\n%s";
            if ($command instanceof CreateCommand
                || $command instanceof UpdateCommand) {
                $dataset = $command->getDataset();
                $sendedValue = $dataset[$propName] ?? null;
                if (!is_scalar($sendedValue)) {
                    $sendedValue = Str::toJson($sendedValue);
                }
            }
            throw new BadRequestException(
                vsprintf($errorMsg, [
                    $propName,
                    $sendedValue ?? 'unknown',
                    $e->getTraceAsString(),
                ]),
                $e
            );
        } catch (\Move\Command\Exception\ConflictException $e) {
            throw new ConflictException($e->getMessage(), $e);
        } catch (OutOfBoundsException $e) {
            throw new BadRequestException($e->getMessage(), $e);
        } catch (BadParameterException $e) {
            throw new BadRequestException($e->getMessage(), $e);
        } catch (RelationException $e) {
            if ($e->getForeignKey()) {
                $msg = 'Relation check failed on column ' . $e->getForeignKey();
            }
            throw new BadRequestException(
                vsprintf($errorMsg, [
                    \get_class($e),
                    $msg ?? $e->getMessage(),
                    $e->getFile(),
                    $e->getLine(),
                    $e->getTraceAsString(),
                ]),
                $e
            );
        } catch (\Move\Command\Exception\NotFoundException $e) {
            throw new NotFoundException($e->getMessage(), $e);
        } catch (\BadMethodCallException $e) {
            throw new BadRequestException($e->getMessage(), $e);
        }

        return $result;
    }
}
