<?php

namespace Cms\Search;

/**
 * Class AbstractSearchClient
 * @package Cms\Search
 */
abstract class AbstractSearchClient implements
    SearchClientSeekableInterface,
    SearchClientAdapterAwareInterface
{
    use SearchClientAdapterAwareTrait;
    use SortableSearchTrait;

    /** @var int */
    protected $offset = 0;

    /** @var int */
    protected $page = 0;

    /** @var int */
    protected $limit = 10;

    /** @var string */
    protected $indexName;

    /** @var  array */
    protected $scroll;

    /** @var  array */
    protected $sourceFields;

    /**
     * ArticleSearchClient constructor.
     * @param SearchClientAdapterInterface $searchClient
     * @param string $indexName
     */
    public function __construct(
        SearchClientAdapterInterface $searchClient,
        string $indexName = null
    ) {
        $this->searchClient = $searchClient;
        $this->indexName = $indexName ?? CmsIndexFactory::INDEX_NAME;
    }

    /**
     * @return string
     */
    abstract protected function getMappingType() : string;

    /**
     * @return array
     */
    abstract protected function getDefaultMatchFields() : array;

    /**
     * @param string|array|ElasticQueryBuilderInterface $query
     * @param array $fields
     * @param array $sortCols
     * @return \Cms\Search\ResultElementInterface[]|ResultSetInterface
     * @throws \InvalidArgumentException
     */
    public function search($query, array $fields = [], array $sortCols = [])
    {
        // ajoute les params de recherche
        $sortParam = $this->parseSortCols($sortCols);
        $opts = [
            'offset' => $this->offset ?: ($this->limit * ($this->page ?: 0)),
            'size' => $this->limit,
            'type' => $this->getMappingType(),
            'sort' => $sortParam,
        ];
        if ($this->sourceFields) {
            $opts['_source'] = $this->sourceFields;
        }
        if (!empty($this->scroll[0])) {
            $opts['options']['scroll'] = $this->scroll[0];
            if (!empty($this->scroll[1])) {
                $opts['options']['scroll_id'] = $this->scroll[1];
            }
        }

        // execution de la query avec le builder
        if ($query instanceof ElasticQueryBuilderInterface) {
            $queryparam = $query->getQueryAsArray();
            if (($aggsData = $query->getAggsAsArray())) {
                $opts['aggs'] = $aggsData;
            }
            if (($sourceData = $query->getSourceFilterAsArray())) {
                $opts['_source'] = $sourceData;
            }
            $results = $this->searchClient->searchArray(
                $this->indexName,
                $queryparam,
                $opts
            );
        } elseif (\is_array($query)) {
            // creation de la query de recherche
            $results = $this->searchClient->searchArray($this->indexName, $query, $opts);
        } elseif (\is_string($query)) {
            // creation de la query de recherche
            $opts['fields'] = $fields ?: $this->getDefaultMatchFields();
            $results = $this->searchClient->searchString($this->indexName, $query, $opts);
        } else {
            throw new \InvalidArgumentException('query must be an array or a string');
        }
        return $results;
    }

    /**
     * @param int $offset
     * @return $this
     */
    public function setOffset(int $offset)
    {
        $this->offset = $offset;
        return $this;
    }

    /**
     * @param int $page
     * @return $this
     */
    public function setPage(int $page)
    {
        $this->page = $page;
        return $this;
    }

    /**
     * @param int $limit
     * @return $this
     */
    public function setLimit(int $limit)
    {
        $this->limit = $limit;
        return $this;
    }

    /**
     * @param string $time
     * @param string|null $scrollId
     * @return $this
     */
    public function useScroll(string $time, string $scrollId = null)
    {
        $this->scroll = [$time, $scrollId];
        return $this;
    }

    /**
     * @param array $sourceFields
     * @return $this
     */
    public function setSourceFields(array $sourceFields)
    {
        $this->sourceFields = $sourceFields;
        return $this;
    }
}
