<?php

namespace Cms\Utils;

use Location\Bounds;
use Location\Coordinate;
use Location\Distance\DistanceInterface;
use Location\Distance\Vincenty;

/**
 * Like vsprintf, but accepts $args keys instead of order index.
 * Both numeric and strings matching /[a-zA-Z0-9_-]+/ are allowed.
 * Example: vskprintf('y = %y$d, x = %x$1.1f', array('x' => 1, 'y' => 2))
 * Result:  'y = 2, x = 1.0'
 * $args also can be object, then it's properties are retrieved
 * using get_object_vars().
 * '%s' without argument name works fine too. Everything vsprintf() can do
 * is supported.
 * @param string $str
 * @param array|mixed $args
 * @return string
 */
function vksprintf($str, $args)
{
    if (\is_object($args)) {
        $args = get_object_vars($args);
    }
    $map = array_flip(array_keys($args));
    $newStr = preg_replace_callback(
        '/(^|[^%])%([a-zA-Z0-9_-]+)\$/',
        function ($m) use ($map) {
            return $m[1] . '%' . ($map[$m[2]] + 1) . '$';
        },
        $str
    );
    return vsprintf($newStr, $args);
}

/**
 * @param array $northEast
 * @param array $southWest
 * @param \Location\Distance\DistanceInterface|null $calculator
 * @return float|int
 */
function geo_diagonal(array $northEast, array $southWest, DistanceInterface $calculator = null)
{
    // distance entre les deux points pour trouver la bonne precision
    $diagonal = (new Coordinate((float)$northEast[0], (float)$northEast[1]))
        ->getDistance(new Coordinate((float)$southWest[0], (float)$southWest[1]), $calculator ?: new Vincenty());
    $diagonal /= 1000;
    return $diagonal;
}

/**
 * @param array $northWest
 * @param array $southEast
 * @return \Location\Coordinate
 */
function geo_center(array $northWest, array $southEast)
{
    return (new Bounds(
        new Coordinate((float)$northWest[1], (float)$northWest[0]),
        new Coordinate((float)$southEast[1], (float)$southEast[0])
    ))->getCenter();
}
